<?php
/**
 * 2007-2016 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 * @author    PrestaShop SA <contact@prestashop.com>
 * @copyright 2007-2016 PrestaShop SA
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

use PrestaShop\PrestaShop\Adapter\Category\CategoryProductSearchProvider;
use PrestaShop\PrestaShop\Adapter\Image\ImageRetriever;
use PrestaShop\PrestaShop\Adapter\Product\PriceFormatter;
use PrestaShop\PrestaShop\Adapter\Product\ProductColorsRetriever;
use PrestaShop\PrestaShop\Core\Product\ProductListingPresenter;
use PrestaShop\PrestaShop\Core\Product\Search\ProductSearchContext;
use PrestaShop\PrestaShop\Core\Product\Search\ProductSearchQuery;
use PrestaShop\PrestaShop\Core\Product\Search\SortOrder;

if (!defined('_PS_VERSION_')) {
    exit;
}

class Afilor extends Module
{
    protected $inputs;
    protected $upload_dir = _PS_UPLOAD_DIR_ . 'afilor_module/';

    protected $messages = [
      UPLOAD_ERR_INI_SIZE   => "Votre fichier est trop gros. La taille maximum pour l'upload est dépassée.",
      UPLOAD_ERR_FORM_SIZE  => "Votre fichier est trop gros.",
      UPLOAD_ERR_PARTIAL    => "L'envoie du fichier à été intérrompue. Le fichier n'est pas parvenu en entier",
      UPLOAD_ERR_NO_FILE    => "Aucun fichier détecté.",
      UPLOAD_ERR_NO_TMP_DIR => "Impossible de trouver un dossier temporaire.",
      UPLOAD_ERR_CANT_WRITE => "Impossible d'enregistrer le fichier sur le serveur.",
      UPLOAD_ERR_EXTENSION  => "Une extension du serveur à bloquée l'envoie du fichier.",
    ];

    public function __construct()
    {
        $this->name          = 'afilor';
        $this->author        = 'webidea - tmo';
        $this->version       = '1.0.0';
        $this->tab           = 'front_office_features';
        $this->need_instance = 0;

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Afilor';
        $this->description = 'Toute la configuration du thème';

        $this->ps_versions_compliancy = ['min' => '1.7', 'max' => _PS_VERSION_];


        $this->inputs = [
          [
            'type'     => 'text',
            'label'    => $this->l('ID Catégorie home page'),
            'name'     => 'categorie_home',
            'size'     => 3,
            'required' => true
          ],
          [
            'type'     => 'text',
            'label'    => $this->l('[ENCART #1] Titre'),
            'name'     => 'afilor_encart_1_titre',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'         => 'textarea',
            'autoload_rte' => true,
            'label'        => $this->l("[ENCART #1] Contenu"),
            'name'         => 'afilor_encart_1_content',
            'size'         => 200,
            'required'     => false
          ],
          [
            'type'     => 'file',
            'label'    => $this->l('[ENCART #1] Photo'),
            'name'     => 'afilor_encart_1_photo1',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'     => 'text',
            'label'    => $this->l('[ENCART #1] Lien'),
            'name'     => 'afilor_encart_1_lien1',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'     => 'text',
            'label'    => $this->l('[ENCART #1] Titre bouton'),
            'name'     => 'afilor_encart_1_bouton1',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'     => 'file',
            'label'    => $this->l('[ENCART #1] Photo 2'),
            'name'     => 'afilor_encart_1_photo2',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'     => 'text',
            'label'    => $this->l('[ENCART #1] Lien 2'),
            'name'     => 'afilor_encart_1_lien2',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'     => 'text',
            'label'    => $this->l('[ENCART #1] Titre bouton 2'),
            'name'     => 'afilor_encart_1_bouton2',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'     => 'text',
            'label'    => $this->l('[ENCART #2] Titre'),
            'name'     => 'afilor_encart_2_titre',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'         => 'textarea',
            'autoload_rte' => true,
            'label'        => $this->l("[ENCART #2] Contenu"),
            'name'         => 'afilor_encart_2_content',
            'size'         => 200,
            'required'     => false
          ],
          [
            'type'     => 'file',
            'label'    => $this->l('[ENCART #2] Photo'),
            'name'     => 'afilor_encart_2_photo1',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'     => 'text',
            'label'    => $this->l('[ENCART #2] Lien'),
            'name'     => 'afilor_encart_2_lien1',
            'size'     => 20,
            'required' => true
          ],
          [
            'type'     => 'text',
            'label'    => $this->l('[ENCART #2] Titre bouton'),
            'name'     => 'afilor_encart_2_bouton1',
            'size'     => 20,
            'required' => true
          ],
        ];
    }

    private function checkDir()
    {
        if (!is_dir($this->upload_dir)) {
            mkdir($this->upload_dir);
        }
    }

    public function install()
    {
        $this->checkDir();
        return parent::install();
    }

    public function uninstall()
    {
        return parent::uninstall();
    }

    public function installTab()
    {
        $tab             = new Tab();
        $tab->active     = 1;
        $tab->class_name = "AfilorTab";
        $tab->name       = [];
        foreach (Language::getLanguages(true) as $lang) {
            $tab->name[$lang['id_lang']] = "Afilor";
        }
        $tab->id_parent = (int)Tab::getIdFromClassName('AdminParentThemes');
        $tab->module    = $this->name;
        return $tab->add();
    }

    public function getContent()
    {
        $output = null;

        if (Tools::isSubmit('submit' . $this->name)) {
            $this->checkDir();

            if (!empty($_FILES)) {
                $allfiles = json_decode(Configuration::get('afilor_files', null, null, null, []), true);
                foreach ($_FILES as $name => $_FILE) {
                    if (isset($_POST[$name]) and !empty($_POST[$name])) {
                        $file = $this->add($_FILE);
                        if (!$file->error) {
                            $allfiles[$name] = '/upload/afilor_module/' . $file->date_file;
                        }
                    }
                }

                Configuration::updateValue('afilor_files', json_encode($allfiles));
            }

            $values = [];
            foreach ($this->inputs as $input) {
                $values[$input['name']] = htmlentities(Tools::getValue($input['name']));
            }
            $AFILORvalues = json_encode($values);

            if (empty($AFILORvalues)) {
                $output .= $this->displayError($this->l('Invalid Configuration value'));
            } else {
                Configuration::updateValue('afilor_values', $AFILORvalues);
                $output .= $this->displayConfirmation($this->l('Settings updated'));
            }
        }
        return $output . $this->displayForm();
    }

    public function displayForm()
    {
        $default_lang = (int)Configuration::get('PS_LANG_DEFAULT');

        $fields_form[0]['form'] = [
          'legend' => [
            'title' => $this->l('Page d\'accueil'),
          ],
          'input'  => $this->inputs,
          'submit' => [
            'title' => $this->l('Save'),
            'class' => 'btn btn-default pull-right'
          ]
        ];

        $helper = new HelperForm();

        // Module, token and currentIndex
        $helper->module          = $this;
        $helper->name_controller = $this->name;
        $helper->token           = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex    = AdminController::$currentIndex . '&configure=' . $this->name;

        // Language
        $helper->default_form_language    = $default_lang;
        $helper->allow_employee_form_lang = $default_lang;

        // Title and toolbar
        $helper->title          = $this->displayName;
        $helper->show_toolbar   = true;        // false -> remove toolbar
        $helper->toolbar_scroll = true;      // yes - > Toolbar is always visible on the top of the screen.
        $helper->submit_action  = 'submit' . $this->name;
        $helper->toolbar_btn    = [
          'save' =>
            [
              'desc' => $this->l('Save'),
              'href' => AdminController::$currentIndex . '&configure=' . $this->name . '&save' . $this->name .
                        '&token=' . Tools::getAdminTokenLite('AdminModules'),
            ],
          'back' => [
            'href' => AdminController::$currentIndex . '&token=' . Tools::getAdminTokenLite('AdminModules'),
            'desc' => $this->l('Back to list')
          ]
        ];

        $AFILORvalues = json_decode(Configuration::get('afilor_values'), true);
        foreach ($AFILORvalues as $key => $val) {
            $AFILORvalues[$key] = html_entity_decode($val);
        }

        $AFILORfiles           = json_decode(Configuration::get('afilor_files', null, null, null, []), true);
        $AFILORvalues['files'] = $AFILORfiles;
        $helper->fields_value  = $AFILORvalues;

        // Load current value
        /*$helper->fields_value['oms_list_encart_2'] = Configuration::get('oms_list_encart_2');
        $helper->fields_value['oms_list_encart_big'] = Configuration::get('oms_list_encart_big');
        $helper->fields_value['oms_astuce'] = Configuration::get('oms_astuce');
        $helper->fields_value['oms_pensee'] = Configuration::get('oms_pensee');*/

        return $helper->generateForm($fields_form);
    }

    public function add($fichier, $params = [])
    {
        $date = date('Yms-His');

        $ext       = '.' . pathinfo($fichier['name'], PATHINFO_EXTENSION);
        $base_name = $this->slugify(strtolower(str_replace($ext, '', $fichier['name'])));

        $params = array_merge([
          'name'   => $base_name,
          'folder' => $this->upload_dir
        ], $params);


        $url = $date . '_' . $this->slugify(strtolower($params['name'])) . $ext;
        //$type = $fichier['type'];
        $error = $fichier['error'];

        if ($error == UPLOAD_ERR_OK) {
            if (!is_dir($params['folder'])) {
                mkdir($params['folder']);
            }

            if (!move_uploaded_file($fichier['tmp_name'], $params['folder'] . $url)) {
                $error = "Erreur lors de l'import du fichier";
            }
        } else {
            $error = isset($this->messages[$error]) ? $this->messages[$error] : "Erreur inconnue";
        }

        return (object)[
          'error'         => $error,
          'original_file' => $fichier['name'],
          'date_file'     => $url
        ];
    }

    public function slugify($text)
    {

        // replace non letter or digits by -
        $text = preg_replace('~[^\\pL\d]+~u', '-', $text);

        // trim
        $text = trim($text, '-');

        // transliterate
        $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);

        // lowercase
        $text = strtolower($text);

        // remove unwanted characters
        $text = preg_replace('~[^-\w]+~', '', $text);

        if (empty($text)) {
            return uniqid();
        }

        return $text;
    }

    public function renderWidget($hookName, array $configuration)
    {
        var_dump($hookName);
        die();
    }

    public function hookDisplayHome()
    {
        $AFILORvalues = json_decode(Configuration::get('afilor_values', null, null, null, []), true);
        $AFILORfiles  = json_decode(Configuration::get('afilor_files', null, null, null, []), true);

        //$categorie = new Category($AFILORvalues['categorie_home'], 1, 1);
        $products = $this->getProducts($AFILORvalues['categorie_home']);

        $this->context->smarty->assign([
          "values"        => $AFILORvalues,
          "files"         => $AFILORfiles,
          "products_home" => $products
        ]);
        return $this->display(__FILE__, 'views/templates/front/hook.tpl');
    }


    protected function getProducts($cat)
    {
        $category = new Category((int)$cat);

        $searchProvider = new CategoryProductSearchProvider(
          $this->context->getTranslator(),
          $category
        );

        $context = new ProductSearchContext($this->context);

        $query = new ProductSearchQuery();

        //$nProducts = Configuration::get('HOME_FEATURED_NBR');
        $nProducts = 4;

        $query
          ->setResultsPerPage($nProducts)
          ->setPage(1)
          ->setSortOrder(new SortOrder('product', 'position', 'asc'));

        $result = $searchProvider->runQuery(
          $context,
          $query
        );

        $assembler = new ProductAssembler($this->context);

        $presenterFactory     = new ProductPresenterFactory($this->context);
        $presentationSettings = $presenterFactory->getPresentationSettings();
        $presenter            = new ProductListingPresenter(
          new ImageRetriever(
            $this->context->link
          ),
          $this->context->link,
          new PriceFormatter(),
          new ProductColorsRetriever(),
          $this->context->getTranslator()
        );

        $products_for_template = [];

        foreach ($result->getProducts() as $rawProduct) {
            $products_for_template[] = $presenter->present(
              $presentationSettings,
              $assembler->assembleProduct($rawProduct),
              $this->context->language
            );
        }

        return $products_for_template;
    }
}
