<?php
/**
 * RelatedProducts Merchandizing (Version 3.0.2)
 *
 * @author    Lineven
 * @copyright 2020 Lineven
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 * International Registered Trademark & Property of Lineven
 */

class LinevenRlpFormAdminOnHookForm extends LinevenRlpForm
{
    private $hook_code;
    private $hook_name;
    private $controller_name;

    /**
     * Constructor.
     * @param string $hook_code Hook code
     * @param string $hook_name Hook name
     * @param string $controller_name Controller name
     *
     * @return void
     */
    public function __construct($hook_code, $hook_name, $controller_name = '')
    {
        $this->hook_code = $hook_code;
        $this->hook_name = $hook_name;
        $this->controller_name = $controller_name;
        parent::__construct();
    }

    /**
     * Get products images types.
     * @return array
     */
    protected function getProductsImagesTypes()
    {
        // Specific settings for product
        $images_products = array();
        foreach (ImageType::getImagesTypes('products') as $image_type_product) {
            $images_products[] = array(
                'id' => $image_type_product['id_image_type'],
                'name' => $image_type_product['name'].
                    ' ('.$image_type_product['width'].'x'.$image_type_product['height'].'px)'
            );
        }
        return $images_products;
    }

    /**
     * Get templates.
     * @return array
     */
    protected function getTemplates()
    {
        // Templates
        $templates = array();
        switch ($this->hook_code) {
            case 'OCL':
                $templates[] = array(
                    'id' => RelatedProducts::$template_standard_classic,
                    'name' => RelatedProducts::$native_templates[RelatedProducts::$template_standard_classic]
                );
                break;
            case 'OQV':
            case 'OAC':
                $templates[] = array(
                    'id' => RelatedProducts::$template_standard_thumbnails,
                    'name' => RelatedProducts::$native_templates[RelatedProducts::$template_standard_thumbnails]
                );
                $templates[] = array(
                    'id' => RelatedProducts::$template_standard_theme,
                    'name' => RelatedProducts::$native_templates[RelatedProducts::$template_standard_theme]
                );
                break;
            case 'OHP':
                $templates[] = array(
                    'id' => RelatedProducts::$template_standard_theme,
                    'name' => RelatedProducts::$native_templates[RelatedProducts::$template_standard_theme]
                );
                break;
            default:
                foreach (RelatedProducts::$native_templates as $code => $name) {
                    $templates[] = array(
                        'id' => $code,
                        'name' => $name
                    );
                }
                break;
        }
        $folder = _LINEVEN_MODULE_RLP_REQUIRE_DIR_.'/views/templates/hook/'.$this->hook_name.'/';
        // Custom templates
        if (is_dir($folder)) {
            $templates_dir = opendir($folder);
            $native_templates = array_change_key_case(RelatedProducts::$native_templates, CASE_LOWER);
            while ($entry = @readdir($templates_dir)) {
                if (!is_dir($folder.$entry) &&
                    $entry != '.' && $entry != '..') {
                    $entry_info = pathinfo($entry);
                    $template_name = basename($entry, '.'.$entry_info['extension']);
                    if (!array_key_exists($template_name, $native_templates) &&
                        Tools::substr($entry, 0, 1) != '_' && $entry_info['extension'] == 'tpl') {
                        $template_specific_name = $template_name.' ('.$this->translator->l('Custom', 'OnHookForm').')';
                        if (Tools::strpos($entry, 'theme')) {
                            $template_specific_name = $template_name.' ('.$this->translator->l('Theme custom', 'OnHookForm').')';
                        }
                        $templates[] = array(
                            'id' => $template_name,
                            'name' => $template_specific_name
                        );
                    }
                }
            }
        }
        return $templates;
    }

    /**
     * Render hook to use fieldset.
     * @param array $description Fieldset description
     * @param array $options Select list options
     *
     * @return array
     */
    protected function renderHookToUseFieldset($description, $options)
    {
        $specific_hook_value = '';
        if ($this->controller_name != '' &&
            Configuration::get('LINEVEN_RLP_SPECIFIC_HOOK_DEF') &&
            Configuration::get('LINEVEN_RLP_SPECIFIC_HOOK_DEF') != '') {
            $hooks_def = unserialize(Configuration::get('LINEVEN_RLP_SPECIFIC_HOOK_DEF'));
            if (is_array($hooks_def)) {
                foreach ($hooks_def as $hook_name => $controller_name) {
                    if ($controller_name == $this->controller_name) {
                        $specific_hook_value = $hook_name;
                        break;
                    }
                }
            }
        }
        $this->fields['hook_settings'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Hook settings', 'OnHookForm'),
                    'icon' => 'fa fa-link'
                ),
                'description_multilines' => $description,
                'input' => array(
                    array(
                        'type' => 'select',
                        'label' => $this->translator->l('Hook to use', 'OnHookForm'),
                        'name' => 'hook_to_use',
                        'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_HOOK_USE',
                        'class' => 'fixed-width-xxl',
                        'options' => array(
                            'query' => $options,
                            'id' => 'id',
                            'name' => 'name'
                        )
                    ),
                    array(
                        'type' => 'html',
                        'name' => 'link_to_hook_confirm',
                        'specific' => 'alert',
                        'alert_type' => 'success',
                        'html_content' => $this->translator->l('The module has been linked to this hook.', 'OnHookForm')
                    ),
                    array(
                        'type' => 'html',
                        'name' => 'link_to_hook_error',
                        'specific' => 'alert',
                        'alert_type' => 'danger',
                        'html_content' => $this->translator->l('An error occured when link module to this hook. Please do it manually in the Prestashop Position menu.', 'OnHookForm')
                    ),
                    array(
                        'type' => 'textbutton',
                        'label' => $this->translator->l('Specific hook to use', 'OnHookForm'),
                        'name' => 'specific_hook_to_use',
                        'maxlength' => 255,
                        'value' => $specific_hook_value,
                        'button' => array(
                            'label' => $this->translator->l('Link the module to this hook', 'OnHookForm'),
                            'attributes' => array(
                                'onclick' => 'LinevenRlpModule.Hooks.linkToSpecificHook();',
                            ),
                        ),
                    )
                ),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'OnHookForm'),
                )
            ),
        );
    }

    /**
     * Render asynchronous settings to fieldset.
     * @param array $description Fieldset description
     * @param array $other_inputs Other inputs
     *
     * @return array
     */
    protected function renderAsynchronousSettingsFieldset($description = array(), $other_inputs = array())
    {
        $description_multilines = array(
            $this->translator->l('Asynchronous mode displays the list of products only after the page has been fully loaded.', 'OnHookForm')
        );
        if ($description) {
            $description_multilines = $description;
        }
        $this->fields['asynchronous_mode'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Asynchronous mode', 'OnHookForm'),
                    'icon' => 'fa fa-refresh'
                ),
                'description_multilines' => $description_multilines,
                'input' => array(
                    array(
                        'type' => 'switch',
                        'label' => $this->translator->l('Asynchronous mode', 'OnHookForm'),
                        'name' => 'is_asynchronous',
                        'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_IS_ASYNC',
                        'is_bool' => true,
                        'values' => array(
                            array(
                                'id' => 'is_asynchronous_on',
                                'value' => 1,
                                'label' => $this->translator->l('Yes', 'OnHookForm')
                            ),
                            array(
                                'id' => 'is_asynchronous_off',
                                'value' => 0,
                                'label' => $this->translator->l('No', 'OnHookForm')
                            )
                        )
                    )
                ),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'OnHookForm'),
                )
            )
        );
        if ($other_inputs) {
            $this->fields['asynchronous_mode']['form']['input'][] = $other_inputs;
        }
    }

    /**
     * Render section activation fieldset.
     * @param string $additional_description Additional description for the fieldset
     * @return array
     */
    protected function renderSectionActivationFieldset($additional_description = null)
    {
        $fieldset_description = array(
            $this->translator->l('If you used several plugins for cross-selling (same category, accessories, etc...), you can display products in separate section for this page.', 'OnHookForm'),
            $this->translator->l('In this case, each activated plugin will have is own section to display products (same category, accessories, etc...).', 'OnHookForm'),
            $this->translator->l('If you choose a single section, all products to display will be grouped in a single section.', 'OnHookForm'),
            $this->translator->l('You can define the section title, the maximum products to display and plugins to use to display products.', 'OnHookForm'),
        );
        if ($additional_description != null) {
            $fieldset_description = array_merge(
                $fieldset_description,
                $additional_description
            );
        }
        $this->fields['sections_activation'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Display sections method', 'OnHookForm'),
                    'icon' => 'fa fa-toggle-on'
                ),
                'description_multilines' => $fieldset_description,
                'input' => array_merge(
                    array(
                        array(
                            'type' => 'switch',
                            'label' => $this->translator->l('Display in separate sections', 'OnHookForm'),
                            'name' => 'separate_display',
                            'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_SEPARATE_DISPLAY',
                            'is_bool' => true,
                            'values' => array(
                                array(
                                    'id' => 'separate_display_on',
                                    'value' => 1,
                                    'label' => $this->translator->l('Yes', 'OnHookForm')
                                ),
                                array(
                                    'id' => 'separate_display_off',
                                    'value' => 0,
                                    'label' => $this->translator->l('No', 'OnHookForm')
                                )
                            ),
                        )
                    ),
                    $this->getOneSectionSettingsFormInput()
                ),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'OnHookForm'),
                )
            ),
        );
    }

    /**
     * Render one section settings fieldset.
     * @return array
     */
    protected function renderOneSectionSettingsFieldset()
    {
        $this->fields['sections_activation'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Section settings', 'OnHookForm'),
                    'icon' => 'fa fa-angle-right'
                ),
                'description_multilines' => array(
                    $this->translator->l('All products to display will be grouped in a single section.', 'OnHookForm'),
                    $this->translator->l('You can define the section title, the maximum products to display and plugins to use to display products.', 'OnHookForm')
                ),
                'input' => $this->getOneSectionSettingsFormInput(),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'OnHookForm'),
                )
            )
        );
    }

    /**
     * Get one section settings form input.
     * @return array
     */
    protected function getOneSectionSettingsFormInput()
    {
        return array(
            array(
                'type' => 'text',
                'label' => $this->translator->l('Section title', 'OnHookForm'),
                'name' => 'section_title',
                'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_TITLE',
                'required' => true,
                'maxlength' => 255,
                'lang' => true,
                'desc' => $this->translator->l('The section title appears to your customers before the list of products.', 'OnHookForm')
            ),
            array(
                'type' => 'text',
                'label' => $this->translator->l('Maximum products to display', 'OnHookForm'),
                'name' => 'max_products',
                'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_MAX_PRODUCTS',
                'required' => true,
                'class' => 'fixed-width-xs',
                'maxlength' => 2,
                'hint' => $this->translator->l('This value is required and must be numeric.', 'OnHookForm').
                    ' '.$this->translator->l('Set to 0 to display all products.', 'OnHookForm'),
                'desc' => $this->translator->l('Maximum product to display. Set 0 to display all products.', 'OnHookForm')
            )
        );
    }

    /**
     * Render section display settings fieldset.
     * @param boolean $display_price Display price
     * @param boolean $display_add_cart_button Display add to cart button
     * @param array $other_inputs Other inputs
     * @return array
     */
    protected function renderDisplaySettingsFieldset(
        $display_price = true,
        $display_add_cart_button = true,
        $other_inputs = array()
    ) {
        $this->fields['display_settings'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Display settings', 'OnHookForm'),
                    'icon' => 'fa fa-eye'
                ),
                'input' => array(),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'OnHookForm'),
                )
            ),
        );
        $this->fields['display_settings']['form']['input'] = array_merge(
            $other_inputs,
            array(
                array(
                    'type' => 'select',
                    'label' => $this->translator->l('Template to use', 'OnHookForm'),
                    'name' => 'template_to_use',
                    'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_DISPLAYING_MODE',
                    'options' => array(
                        'query' => $this->getTemplates(),
                        'id' => 'id',
                        'name' => 'name'
                    ),
                    'desc' => $this->translator->l('Choose the most suitable template for your theme.', 'OnHookForm'),
                ),
                array(
                    'type' => 'select',
                    'label' => $this->translator->l('Image type to use', 'OnHookForm'),
                    'name' => 'image_type',
                    'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_IMAGE_TYPE',
                    'options' => array(
                        'query' => $this->getProductsImagesTypes(),
                        'id' => 'id',
                        'name' => 'name'
                    )
                )
            )
        );
        if ($display_price && $this->hook_code != 'OHP') {
            $this->fields['display_settings']['form']['input'][] = array(
                'type' => 'switch',
                'label' => $this->translator->l('Display price', 'OnHookForm'),
                'name' => 'display_price',
                'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_DISPLAY_PRICE',
                'is_bool' => true,
                'values' => array(
                    array(
                        'id' => 'display_price_on',
                        'value' => 1,
                        'label' => $this->translator->l('Yes', 'OnHookForm')
                    ),
                    array(
                        'id' => 'display_price_off',
                        'value' => 0,
                        'label' => $this->translator->l('No', 'OnHookForm')
                    )
                )
            );
        }
        if ($display_add_cart_button && $this->hook_code != 'OHP') {
            $this->fields['display_settings']['form']['input'][] = array(
                'type' => 'switch',
                'label' => $this->translator->l('Display "Add to cart" button', 'OnHookForm'),
                'name' => 'display_cart',
                'configuration_name' => 'LINEVEN_RLP_' . $this->hook_code . '_DISPLAY_CART',
                'is_bool' => true,
                'values' => array(
                    array(
                        'id' => 'display_cart_on',
                        'value' => 1,
                        'label' => $this->translator->l('Yes', 'OnHookForm')
                    ),
                    array(
                        'id' => 'display_cart_off',
                        'value' => 0,
                        'label' => $this->translator->l('No', 'OnHookForm')
                    )
                )
            );
        }
        if ($this->hook_code != 'OHP') {
            $this->fields['display_settings']['form']['input'][] = array(
                'type' => 'switch',
                'label' => $this->translator->l('Display the label "new product"', 'OnHookForm'),
                'name' => 'display_new_product',
                'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_NEW_PRD',
                'is_bool' => true,
                'values' => array(
                    array(
                        'id' => 'display_new_product_on',
                        'value' => 1,
                        'label' => $this->translator->l('Yes', 'OnHookForm')
                    ),
                    array(
                        'id' => 'display_new_product_off',
                        'value' => 0,
                        'label' => $this->translator->l('No', 'OnHookForm')
                    )
                )
            );
            $this->fields['display_settings']['form']['input'][] = array(
                'type' => 'switch',
                'label' => $this->translator->l('Display percent reduction', 'OnHookForm'),
                'name' => 'display_reduction',
                'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_DP_REDUCTION',
                'is_bool' => true,
                'values' => array(
                    array(
                        'id' => 'display_reduction_on',
                        'value' => 1,
                        'label' => $this->translator->l('Yes', 'OnHookForm')
                    ),
                    array(
                        'id' => 'display_reduction_off',
                        'value' => 0,
                        'label' => $this->translator->l('No', 'OnHookForm')
                    )
                )
            );
        }

        if ($this->hook_code != 'OHP' && $this->hook_code != 'OCL') {
            $this->fields['display_thumbnails_settings'] = array(
                'form' => array(
                    'legend' => array(
                        'title' => $this->translator->l('Thumbnails settings', 'OnHookForm'),
                        'icon' => 'fa fa-picture-o'
                    ),
                    'description' => $this->translator->l('Related settings only for the thumbnails mode.', 'OnHookForm'),
                    'input' => array(
                        array(
                            'type' => 'switch',
                            'label' => $this->translator->l('Display title', 'OnHookForm'),
                            'name' => 'display_title',
                            'configuration_name' => 'LINEVEN_RLP_' . $this->hook_code . '_THUMB_DP_TITLE',
                            'is_bool' => true,
                            'values' => array(
                                array(
                                    'id' => 'display_title_on',
                                    'value' => 1,
                                    'label' => $this->translator->l('Yes', 'OnHookForm')
                                ),
                                array(
                                    'id' => 'display_title_off',
                                    'value' => 0,
                                    'label' => $this->translator->l('No', 'OnHookForm')
                                )
                            )
                        ),
                        array(
                            'type' => 'switch',
                            'label' => $this->translator->l('Display description', 'OnHookForm'),
                            'name' => 'display_description',
                            'configuration_name' => 'LINEVEN_RLP_' . $this->hook_code . '_THUMB_DP_DESC',
                            'is_bool' => true,
                            'values' => array(
                                array(
                                    'id' => 'display_description_on',
                                    'value' => 1,
                                    'label' => $this->translator->l('Yes', 'OnHookForm')
                                ),
                                array(
                                    'id' => 'display_description_off',
                                    'value' => 0,
                                    'label' => $this->translator->l('No', 'OnHookForm')
                                )
                            )
                        ),
                        array(
                            'type' => 'text',
                            'label' => $this->translator->l('Height block description', 'OnHookForm'),
                            'name' => 'description_height',
                            'configuration_name' => 'LINEVEN_RLP_' . $this->hook_code . '_THUMB_DP_HEIGHT',
                            'required' => true,
                            'suffix' => 'px',
                            'class' => 'fixed-width-xs',
                            'maxlength' => 3,
                            'hint' => $this->translator->l('This value is required and must be numeric and positive.', 'OnHookForm'),
                            'desc' => $this->translator->l('This value determines the height of the description block (title and/or description).', 'OnHookForm')
                        )
                    ),
                    'submit' => array(
                        'title' => $this->translator->l('Save', 'OnHookForm'),
                    )
                ),
            );
        }
    }

    /**
     * Render plugins fieldset.
     * @return array
     */
    protected function renderPluginsFieldset()
    {
        $this->fields['plugins'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Plugins', 'OnHookForm'),
                    'icon' => 'fa fa-plug'
                ),
                'description_multilines' => array(
                    $this->translator->l('For all activated plugins, you can choose plugins to use for this hook.', 'OnHookForm'),
                    $this->translator->l('If you display products lists in several sections, you can define the maximum products to display for each plugin.', 'OnHookForm'),
                    '',
                    $this->translator->l('Be careful, think about your server resources. Select and use plugins with relevance. Each plugin requires the execution of several queries to find and display the products.', 'OnHookForm')
                ),
                'input' => array(
                    array(
                        'type' => 'html',
                        'html_content' => '',
                        'name' => 'html_fieldset_plugins_content',
                    )
                ),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'OnHookForm'),
                )
            )
        );
    }

    /**
     * Render sort fieldset.
     * @return array
     */
    protected function renderSortSettingsFieldset()
    {

        $this->fields['sort_display_settings'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Sort display settings', 'OnHookForm'),
                    'icon' => 'fa fa-sort'
                ),
                'description' => $this->translator->l('When products to display are found by the module, you can choose the sort method to present products to your customer.', 'OnHookForm'),
                'input' => array(
                    array(
                        'type' => 'select',
                        'label' => $this->translator->l('Sort method for display products in list', 'OnHookForm'),
                        'name' => 'sort_display_method',
                        'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_SORT_DISPLAY',
                        'class' => ' fixed-width-xxl',
                        'options' => array(
                            'query' => array(
                                array(
                                    'id' => RelatedProducts::$sort_display_default,
                                    'name' => $this->translator->l('In the order found', 'OnHookForm')
                                ),
                                array(
                                    'id' => RelatedProducts::$sort_display_name,
                                    'name' => $this->translator->l('By name', 'OnHookForm')
                                ),
                                array(
                                    'id' => RelatedProducts::$sort_display_price,
                                    'name' => $this->translator->l('By price', 'OnHookForm')
                                ),
                                array(
                                    'id' => RelatedProducts::$sort_display_random,
                                    'name' => $this->translator->l('Random', 'OnHookForm')
                                )
                            ),
                            'id' => 'id',
                            'name' => 'name'
                        )
                    ),
                    array(
                        'type' => 'select',
                        'label' => $this->translator->l('Sort way', 'OnHookForm'),
                        'name' => 'sort_display_way',
                        'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_SORT_DISPLAY_WAY',
                        'options' => array(
                            'query' => array(
                                array(
                                    'id' => RelatedProducts::$sort_way_asc,
                                    'name' => $this->translator->l('Ascendant', 'OnHookForm')
                                ),
                                array(
                                    'id' => RelatedProducts::$sort_way_desc,
                                    'name' => $this->translator->l('Descendant', 'OnHookForm')
                                )
                            ),
                            'id' => 'id',
                            'name' => 'name'
                        )
                    )
                ),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'OnHookForm'),
                )
            ),
        );
        $this->fields['research_depth'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Research depth', 'OnHookForm'),
                    'icon' => 'fa fa-search'
                ),
                'description_multilines' => array(
                    $this->translator->l('By default, the module search products in the order of your plugins and, for associations, in the order of associations.', 'OnHookForm'),
                    $this->translator->l('You can choose to random the search between plugins and associations.', 'OnHookForm'),
                    '',
                    $this->translator->l('Moreover, in order not to always offer the same products, you can choose to random your products catalog before products search.', 'OnHookForm'),
                    '',
                    $this->translator->l('Be careful, think about your server resources if you have a large products catalog. Do some tests before activate those feature in production.', 'OnHookForm')
                ),
                'input' => array(
                    array(
                        'type' => 'checkbox',
                        'name' => 'randomize_plugins',
                        'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_RANDOM_PLUGINS',
                        'values' => array(
                            'query' => array(
                                array(
                                    'id' => 'on',
                                    'name' => $this->translator->l('Ignore sort configured for plugins and associations and execute them randomly.', 'OnHookForm'),
                                    'val' => '1'
                                ),
                            ),
                            'id' => 'id',
                            'name' => 'name'
                        ),
                    ),
                    array(
                        'type' => 'checkbox',
                        'name' => 'randomize_search',
                        'configuration_name' => 'LINEVEN_RLP_'.$this->hook_code.'_RANDOM_SEARCH',
                        'values' => array(
                            'query' => array(
                                array(
                                    'id' => 'on',
                                    'name' => $this->translator->l('Randomize the products catalog before search.', 'OnHookForm'),
                                    'val' => '1'
                                ),
                            ),
                            'id' => 'id',
                            'name' => 'name'
                        ),
                    )
                ),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'OnHookForm'),
                )
            ),
        );
    }

    /**
     * Validate.
     * @return boolean
     */
    public function validate()
    {
        $module_context = LinevenRlpContext::getContext();
        $languages = Language::getLanguages(true);

        if (Tools::isSubmit('hook_to_use') &&
            Tools::getValue('hook_to_use') == RelatedProducts::$hook_to_use_specific &&
            Tools::strlen(trim(Tools::getValue('specific_hook_to_use'))) == 0) {
            $this->addError(
                $this->translator->l('You must fill the specific hook name to use by the module.', 'OnHookForm')
            );
        }

        if (Tools::strlen(trim(Tools::getValue('section_title_'.$module_context->default_id_lang))) == 0) {
            $this->addError($this->translator->l('You must enter the section title for your default language (for one section display).', 'OnHookForm'));
        }
        if (Tools::strlen(trim(Tools::getValue('max_products'))) == 0
            || !Validate::isInt(Tools::getValue('max_products'))) {
            $this->addError(
                $this->translator->l('You must fill a correct maximum products value (for one section display).', 'OnHookForm')
            );
        }
        if (Tools::isSubmit('description_height') &&
            (Tools::strlen(trim(Tools::getValue('description_height'))) == 0
                || !Validate::isInt(Tools::getValue('description_height')))) {
            $this->addError(
                $this->translator->l('You must fill a correct height block description (for thumbnails template).', 'OnHookForm')
            );
        }
        $count_section = 0;
        if (Tools::isSubmit('hook_plugins_positions') && Tools::getValue('hook_plugins_positions') != '') {
            $plugins_ids = explode(',', Tools::getValue('hook_plugins_positions'));
            foreach ($plugins_ids as $id_plugin) {
                if (Tools::isSubmit('hook_plugin_id_'.$id_plugin) && Tools::isSubmit('hook_max_products_plugin_'.$id_plugin)) {
                    $count_section++;
                    if (Tools::strlen(trim(Tools::getValue('hook_max_products_plugin_'.$id_plugin))) == 0
                        || !Validate::isInt(Tools::getValue('hook_max_products_plugin_'.$id_plugin))) {
                        $this->addError(
                            $this->translator->l('You must fill a correct maximum products value for plugins.', 'OnHookForm')
                        );
                        break;
                    }
                }
            }
        }
        if ($count_section == 0) {
            $this->addError(
                $this->translator->l('You must select at least one plugin to use.', 'OnHookForm')
            );
        }

        // Check for each language
        foreach ($languages as $language) {
            $language_name = $language['name'];
            if (Tools::strlen(Tools::getValue('section_title_'.$language['id_lang'])) > 255) {
                $this->addError(
                    $this->translator->l('The section title is too long for the language:', 'OnHookForm').' '.$language_name.'.'
                );
            }
        }
        return parent::validate();
    }

    /**
     * Update settings
     *
     * @return void
     */
    public function updateSettings()
    {
        parent::updateSettings();
        $configuration = LinevenRlpConfiguration::getConfiguration();

        // Save image definitions
        if (Tools::isSubmit('image_type')) {
            $image_types = ImageType::getImagesTypes();
            foreach ($image_types as $type) {
                if ($type['id_image_type'] == (int)Tools::getValue('image_type')) {
                    Configuration::updateValue('LINEVEN_RLP_'.$this->hook_code.'_IMAGE_TYPE_NAME', $type['name']);
                    Configuration::updateValue('LINEVEN_RLP_'.$this->hook_code.'_IMAGE_TYPE_WIDTH', $type['width']);
                    Configuration::updateValue('LINEVEN_RLP_'.$this->hook_code.'_IMAGE_TYPE_HEIGHT', $type['height']);
                    break;
                }
            }
        }

        // Specific hook to use
        $hooks_def = array();
        if (Configuration::get('LINEVEN_RLP_SPECIFIC_HOOK_DEF') &&
            Configuration::get('LINEVEN_RLP_SPECIFIC_HOOK_DEF') != '') {
            $hooks_def = unserialize(Configuration::get('LINEVEN_RLP_SPECIFIC_HOOK_DEF'));
        }
        if (Tools::getValue('hook_to_use') == RelatedProducts::$hook_to_use_specific) {
            if (Tools::getValue('specific_hook_to_use') != '') {
                foreach ($hooks_def as $hook_name => $controller_name) {
                    if ($controller_name == $this->controller_name &&
                        $hook_name != Tools::getValue('specific_hook_to_use')) {
                        $configuration->getModule()->unregisterHook($hook_name);
                        unset($hooks_def[$hook_name]);
                        break;
                    }
                }
                $hooks_def[Tools::getValue('specific_hook_to_use')] = $this->controller_name;
                Configuration::updateValue('LINEVEN_RLP_SPECIFIC_HOOK_DEF', serialize($hooks_def));
            }
        } else {
            foreach ($hooks_def as $hook_name => $controller_name) {
                if ($controller_name == $this->controller_name) {
                    $configuration->getModule()->unregisterHook($hook_name);
                    unset($hooks_def[$hook_name]);
                    Configuration::updateValue('LINEVEN_RLP_SPECIFIC_HOOK_DEF', serialize($hooks_def));
                    break;
                }
            }
        }

        // Plugins
        $values_saved = array();
        if (Tools::isSubmit('hook_plugins_positions') && Tools::getValue('hook_plugins_positions') != '') {
            $plugins_ids = explode(',', Tools::getValue('hook_plugins_positions'));
            $max_products = 5;
            if (!Tools::isSubmit('separate_display') ||
                (Tools::isSubmit('separate_display') && (int)Tools::getValue('separate_display') == 0)) {
                $max_products = Tools::getValue('max_products');
            }
            foreach ($plugins_ids as $id_plugin) {
                if (Tools::isSubmit('hook_plugin_id_'.$id_plugin)) {
                    if (Tools::isSubmit('hook_max_products_plugin_'.$id_plugin) &&
                        Tools::isSubmit('separate_display') && (int)Tools::getValue('separate_display') == 1) {
                        $max_products = (int)Tools::getValue('hook_max_products_plugin_'.$id_plugin);
                    }
                    if (array_key_exists(Tools::getValue('hook_plugin_id_'.$id_plugin), $values_saved) === false) {
                        $plugin_object = new LinevenRlpPluginObject($id_plugin);
                        $plugin_class = $plugin_object->getClass();
                        $plugin_options = array();
                        if ($plugin_class->hasOptions()) {
                            if ($plugin_object->hasOptions()) {
                                $plugin_options = $plugin_object->getOptions();
                                $plugin_form = new LinevenRlpFormAdminPluginOptions($plugin_object, $plugin_class);
                                $plugin_form->populateOptions($plugin_options);
                                $plugin_form->populateFromRequest();
                                $plugin_options_form_values = $plugin_form->getFieldsValue();
                                foreach ($plugin_options as $option_name => $option_value) {
                                    if (isset($plugin_options_form_values[$id_plugin.'_'.$option_name])) {
                                        $option_value = $plugin_options_form_values[$id_plugin.'_'.$option_name];
                                    }
                                    $plugin_options[$option_name] = $option_value;
                                }
                            }
                        }
                        $values_saved[Tools::getValue('hook_plugin_code_'.$id_plugin)] = array(
                            'max_products' => $max_products,
                            'options' => $plugin_options
                        );
                    }
                }
            }
        }
        if (count($values_saved)) {
            Configuration::updateValue('LINEVEN_RLP_'.$this->hook_code.'_PLUGINS', serialize($values_saved));
        }
    }
}
