<?php
/**
 * RelatedProducts Merchandizing (Version 3.0.2)
 *
 * @author    Lineven
 * @copyright 2020 Lineven
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 * International Registered Trademark & Property of Lineven
 */

class LinevenRlpFormAdminPlugin extends LinevenRlpForm
{
    private $plugin;                    // Plugin
    private $plugin_class;              // Plugin class
    private $plugin_description;        // Plugin description

    /**
     * LinevenRlpFormAdminPlugin constructor.
     * @param object $plugin Plugin
     */
    public function __construct($plugin)
    {
        $this->plugin = $plugin;
        $this->plugin_description = '';
        if ((int)$this->plugin->id_plugin != 0) {
            $this->plugin_class = LinevenRlpPluginObject::getStaticClass($this->plugin->classname);
            if ($this->plugin_class != null) {
                $this->plugin_description = $this->plugin_class->getAdminFormDescription();
            }
        }
        parent::__construct();
    }

    /**
     * Init.
     * @return void
     */
    public function init()
    {
        $this->fields['identification'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Plugin class', 'Plugin'),
                    'icon' => 'fa fa-info'
                ),
                'description_multilines' => array(
                    $this->translator->l('This is your own plugin.', 'Plugin')
                ),
                'input' => array(
                    array(
                        'type' => 'hidden',
                        'name' => 'id_plugin'
                    ),
                    array(
                        'type' => 'text',
                        'label' => $this->translator->l('Name', 'Plugin'),
                        'name' => 'name',
                        'required' => true,
                        'maxlength' => 255,
                        'lang' => true,
                        'desc' => $this->translator->l('The name of your plugin will not displayed for your customers.', 'Plugin')
                    )
                ),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'Plugin'),
                )
            )
        );

        if ($this->plugin->type == LinevenRlpPluginObject::$plugin_type_system) {
            $this->fields['identification']['form']['description_multilines'] = array(
                $this->translator->l('This is a plugin system. You can not change identifications informations.', 'Plugin')
            );
        } else {
            $this->fields['identification']['form']['input'][] =
                array(
                    'type' => 'text',
                    'label' => $this->translator->l('Class name', 'Plugin'),
                    'name' => 'classname',
                    'required' => true,
                    'maxlength' => 255,
                    'desc' => $this->translator->l('The class name of your plugin class.', 'Plugin')
                );
        }
        if (is_array($this->plugin_description) && count($this->plugin_description)) {
            $this->fields['identification']['form']['description_multilines'] = array_merge(
                $this->fields['identification']['form']['description_multilines'],
                array(''),
                $this->plugin_description
            );
        } else {
            if ($this->plugin_description != '') {
                $this->fields['identification']['form']['description_multilines'] = array_merge(
                    $this->fields['identification']['form']['description_multilines'],
                    array(
                        '',
                        $this->plugin_description
                    )
                );
            }
        }
        $this->fields['identification']['form']['input'][] =
            array(
                'type' => 'switch',
                'label' => $this->translator->l('Active', 'Plugin'),
                'name' => 'is_active',
                'is_bool' => true,
                'values' => array(
                    array(
                        'id' => 'is_active_on',
                        'value' => 1,
                        'label' => $this->translator->l('Yes', 'Plugin')
                    ),
                    array(
                        'id' => 'is_active_off',
                        'value' => 0,
                        'label' => $this->translator->l('No', 'Plugin')
                    )
                )
            );
        if ($this->plugin_class != null) {
            $admin_plugin_options = $this->plugin_class->getAdminFormOptions();
            if ($admin_plugin_options) {
                $this->fields['options'] = array(
                    'form' => array(
                        'legend' => array(
                            'title' => $this->translator->l('Options', 'Plugin'),
                            'icon' => 'fa fa-cog'
                        ),
                        'input' => $admin_plugin_options,
                        'submit' => array(
                            'title' => $this->translator->l('Save', 'Plugin'),
                        )
                    )
                );
            }
        }
        $this->fields['texts'] = array(
            'form' => array(
                'legend' => array(
                    'title' => $this->translator->l('Texts', 'Plugin'),
                    'icon' => 'fa fa-align-left'
                ),
                'description' => $this->translator->l('If you have activate separate display in dedicated sections, you can define the title for this plugin.', 'Plugin'),
                'input' => array(
                    array(
                        'type' => 'text',
                        'label' => $this->translator->l('Section title', 'Plugin'),
                        'name' => 'title',
                        'required' => false,
                        'maxlength' => 255,
                        'lang' => true,
                        'desc' => $this->translator->l('The section title appears to your customers before the list of products.', 'Plugin')
                    )
                ),
                'submit' => array(
                    'title' => $this->translator->l('Save', 'Plugin'),
                )
            ),
        );
    }

    /**
     * Validate.
     * @return boolean
     */
    public function validate()
    {
        $module_context = LinevenRlpContext::getContext();
        $languages = Language::getLanguages(true);
        if (Tools::strlen(trim(Tools::getValue('name_'.$module_context->default_id_lang))) == 0) {
            $this->addError($this->translator->l('You must enter the name for your default language.', 'Plugin'));
        }
        if ($this->plugin->type != LinevenRlpPluginObject::$plugin_type_system) {
            if (Tools::strlen(trim(Tools::getValue('classname'))) == 0) {
                $this->addError($this->translator->l('You must enter the classname for your plugin.', 'Plugin'));
            }
        }

        if (Tools::strlen(trim(Tools::getValue('title_'.$module_context->default_id_lang))) == 0) {
            $this->addError($this->translator->l('You must enter the section title for your default language.', 'Plugin'));
        }

        foreach ($languages as $language) {
            $language_name = $language['name'];
            if (Tools::strlen(Tools::getValue('name_'.$language['id_lang'])) > 255) {
                $this->addError(
                    $this->translator->l('The name is too long for the language:', 'Plugin').' '.$language_name.'.'
                );
            }
            if (Tools::strlen(Tools::getValue('title_'.$language['id_lang'])) > 255) {
                $this->addError(
                    $this->translator->l('The section title is too long for the language:', 'Plugin').' '.$language_name.'.'
                );
            }
        }
        
        return parent::validate();
    }

    /**
     * Populate.
     *
     * @param Object $object
     * @return void
     */
    public function populate($object)
    {
        parent::populate($object);
        if ($this->plugin_class != null) {
            if ($this->plugin_class->hasOptions()) {
                foreach ($this->plugin_class->getPluginOptions() as $option_name => $option_value) {
                    if ($this->plugin->hasOption($option_name)) {
                        $option_value = $this->plugin->getOption($option_name);
                    }
                    $this->setValue($option_name, $option_value);
                }
            }
        }
    }
}
