<?php
/**
 * RelatedProducts Merchandizing (Version 3.0.2)
 *
 * @author    Lineven
 * @copyright 2020 Lineven
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 * International Registered Trademark & Property of Lineven
 */

class LinevenRlpPluginObject extends LinevenRlpModel
{
    /**
     * Id
     * @var string
     */
    public $id_plugin;

    /**
     * Code
     * @var string
     */
    public $code;

    /**
     * Type
     * @var string
     */
    public $type;

    /**
     * Name
     * @var string
     */
    public $name;

    /**
     * Classname
     * @var string
     */
    public $classname;

    /**
     * Title
     * @var string
     */
    public $title;

    /**
     * Is active
     * @var int
     */
    public $is_active;

    /**
     * Options
     * @var string
     */
    public $options;

    /**
     * Order display
     * @var integer
     */
    public $order_display;

    public static $plugin_type_system   = 'SYS';        /* System */
    public static $plugin_type_user     = 'USR';        /* User */

    public static $definition = array(
        'table' => 'lineven_rlp_plugins',
        'primary' => 'id_plugin',
        'multilang' => true,
        'multilang_shop' => true,
        'fields' => array(
            'code' => array(
                'type' => self::TYPE_STRING,
                'validate' => 'isString',
                'required' => false
            ),
            'type' => array(
                'type' => self::TYPE_STRING,
                'validate' => 'isString',
                'required' => false
            ),
            'name' => array(
                'type' => self::TYPE_NOTHING,
                'required' => false
            ),
            'classname' => array(
                'type' => self::TYPE_STRING,
                'validate' => 'isString',
                'required' => false
            ),
            'title' => array(
                'type' => self::TYPE_STRING,
                'lang' => true,
                'validate' => 'isString',
                'required' => false
            ),
            'is_active' => array(
                'type' => self::TYPE_INT,
                'validate' => 'isUnsignedId',
                'required' => true
            ),
            'options' => array(
                'type' => self::TYPE_NOTHING,
                'required' => false
            ),
            'order_display' => array(
                'type' => self::TYPE_INT,
                'validate' => 'isInt',
                'required' => false
            )
        )
    );

    /**
     * Constructor.
     * @param int $id Identifier
     */
    public function __construct($id = null)
    {
        parent::__construct($id);
        if ($id) {
            $this->name = unserialize($this->name);
            $this->options = unserialize($this->options);
        } else {
            $this->code = md5(time());
            $this->name = array();
            $this->options = array();
            $this->type = self::$plugin_type_user;
            $this->is_active = 0;
            $this->order_display = self::getNewPositionOrder();
        }
    }

    /**
     * Get fields.
     * @return array
     */
    public function getFields()
    {
        $fields = parent::getFields();
        $fields['name'] = serialize($this->name);
        $fields['options'] = serialize($this->options);
        return $fields;
    }

    /**
     * Deletes current object from database
     *
     * @return bool True if delete was successful
     * @throws PrestaShopException
     */
    public function delete()
    {
        $return = parent::delete();
        self::reorder();
        return $return;
    }

    /**
     * Toggle activation.
     * @param int $id Id schedule
     * @return boolean
     */
    public static function toggleActivation($id)
    {
        if ($id != null) {
            // Update
            $sql = 'SELECT `is_active` FROM `'._DB_PREFIX_.'lineven_rlp_plugins` where id_plugin = '.(int)$id;
            $result = Db::getInstance()->ExecuteS($sql);
            if (count($result)) {
                $is_active = !$result[0]['is_active'];
            }
            Db::getInstance()->Execute(
                'UPDATE `'._DB_PREFIX_.'lineven_rlp_plugins`
                SET
                `is_active` = '.(int)$is_active.'
                WHERE id_plugin = '.(int)$id
            );
        }
        return $is_active;
    }

    /**
     * Change order fields.
     * @param array $positions Positions
     * @param string $position_first First position
     * @return boolean
     */
    public static function order($positions, $position_first)
    {
        $position = $position_first;
        $positions = explode(',', $positions);
        if (is_array($positions)) {
            foreach ($positions as $id) {
                // Update
                Db::getInstance()->Execute(
                    'UPDATE `'._DB_PREFIX_.'lineven_rlp_plugins`
                    SET `order_display` = '.(int)$position.'
                    WHERE id_plugin = '.(int)$id
                );
                $position++;
            }
        }
        return true;
    }

    /**
     * Reorder fields.
     * return @void
     */
    public static function reorder()
    {
        $sql = 'SELECT `id_plugin`, `order_display` FROM `'._DB_PREFIX_.'lineven_rlp_plugins` ORDER BY `order_display` ASC, `id_plugin` ASC';
        $list = Db::getInstance()->ExecuteS($sql);
        $order = 1;
        for ($i = 0; $i < count($list); $i++) {
            Db::getInstance()->Execute(
                'UPDATE `'._DB_PREFIX_.'lineven_rlp_plugins`
                SET `order_display` = '.$order.'
                WHERE id_plugin = '.(int)$list[$i]['id_plugin']
            );
            $order++;
        }
    }

    /**
     * Get new position order.
     * return @void
     */
    public static function getNewPositionOrder()
    {
        $sql = 'SELECT count(`id_plugin`) as count FROM `'._DB_PREFIX_.'lineven_rlp_plugins`';
        $list = Db::getInstance()->ExecuteS($sql);
        if ($list) {
            return (int)$list[0]['count']+1;
        }
        return 1;
    }

    /**
     * Get plugins.
     *
     * @param boolean $active Only active
     * @param boolean $full Full informations
     * return @void
     */
    public static function getPlugins($active = true, $full = false)
    {
        $sql = 'SELECT plg.* '.($full ? ', plgl.id_shop, plgl.id_lang, plgl.title ' : '').' FROM `'._DB_PREFIX_.'lineven_rlp_plugins` plg'.
            ($full ?
                ' LEFT JOIN  `'._DB_PREFIX_.'lineven_rlp_plugins_lang` plgl ON (
                    plgl.`id_plugin` = plg.`id_plugin` AND plgl.`id_shop` = '.(int)Context::getContext()->shop->getContextShopID().') '
                : ''
            ).
            ($active ? ' WHERE `is_active` = 1 ' : '').'
            ORDER BY plg.`order_display` ASC, plg.`id_plugin` ASC';
        return Db::getInstance()->ExecuteS($sql);
    }

    /**
     * Is plugin exists.
     *
     * @param boolean $active Only active
     * @param boolean $full Full informations
     * return @void
     */
    public static function isPluginExists($plugin_code)
    {
        $sql = 'SELECT * FROM `'._DB_PREFIX_.'lineven_rlp_plugins` plg
            WHERE `code` = "'.$plugin_code.'"';
        return (count(Db::getInstance()->ExecuteS($sql)) ? true : false);
    }

    /**
     * Get displayed name.
     * @return string
     */
    public function getDisplayedName()
    {
        return self::getStaticDisplayedName($this->name);
    }

    /**
     * Get displayed name.
     *
     * @param array $name Name
     * @return string
     */
    public static function getStaticDisplayedName($name)
    {
        return LinevenRlpTranslator::getTranslationFromArray($name);
    }

    /**
     * Get displayed title.
     * @return string
     */
    public function getDisplayedTitle()
    {
        return self::getStaticDisplayedTitle($this->title);
    }

    /**
     * Get displayed title.
     *
     * @param array $title Title
     * @return string
     */
    public static function getStaticDisplayedTitle($title)
    {
        return LinevenRlpTranslator::getTranslationFromArray($title);
    }

    /**
     * Set options.
     * @param array $options Options
     * @return void
     */
    public function setOptions($options)
    {
        if (is_array($options)) {
            $this->options = $options;
        }
    }

    /**
     * Set option.
     * @param string $name Name
     * @param string $value Value
     * @return void
     */
    public function setOption($name, $value)
    {
        $this->options[$name] = $value;
    }

    /**
     * Get option.
     * @param string $name Name
     * @return mixed
     */
    public function getOption($name)
    {
        if (isset($this->options[$name])) {
            return $this->options[$name];
        }
        return false;
    }

    /**
     * Get options.
     * @return array
     */
    public function getOptions()
    {
        return $this->options;
    }

    /**
     * Has option.
     * @param string $name Name
     * @return boolean
     */
    public function hasOption($name)
    {
        if (isset($this->options[$name])) {
            return true;
        }
        return false;
    }

    /**
     * Has options.
     * @return boolean
     */
    public function hasOptions()
    {
        if (is_array($this->options)) {
            return (count($this->options) ? true : false);
        }
        return false;
    }

    /**
     * Get php file to call.
     * @return boolean|string
     */
    public function getFileToCall()
    {
        return self::getStaticFileToCall($this->classname);
    }

    /**
     * Get php file to call.
     * @param string $classname Classname
     * @return boolean|string
     */
    public static function getStaticFileToCall($classname)
    {
        $folder = 'system';
        if (strpos($classname, 'User')) {
            $folder = 'user';
        }
        $file = _LINEVEN_MODULE_RLP_REQUIRE_DIR_.'/classes/plugin/'.$folder.'/'.Tools::substr($classname, 10).'.php';
        if (is_file($file)) {
            return $file;
        }
        return false;
    }

    /**
     * Get plugin class.
     * @return boolean|string
     */
    public function getClass()
    {
        return self::getStaticClass($this->classname);
    }

    /**
     * Get plugin class .
     * @param string $classname Classname
     * @return boolean|string
     */
    public static function getStaticClass($classname)
    {
        // Get classname
        if (!in_array($classname, get_declared_classes())) {
            $class_file = LinevenRlpPluginObject::getStaticFileToCall($classname);
            if ($class_file !== false) {
                require_once($class_file);
            }
        }
        if (in_array($classname, get_declared_classes())) {
            return new $classname();
        }
        return null;
    }

    /**
     * Fill an object with given data. Data must be an array with this syntax:
     * array(objProperty => value, objProperty2 => value, etc.).
     *
     * @since 1.5.0.1
     *
     * @param array $data
     * @param int|null $id_lang
     */
    public function hydrate(array $data, $id_lang = null)
    {
        parent::hydrate($data, $id_lang);
        $this->name = unserialize($this->name);
        $this->options = unserialize($this->options);
    }

    /**
     * Populate from form.
     *
     * @param Array $array Form value
     * @return void
     */
    public function populateFromFormValues($form_values)
    {
        parent::populateFromFormValues($form_values);
        $plugin_class = LinevenRlpPluginObject::getStaticClass($this->classname);
        if ($plugin_class) {
            if ($plugin_class->hasOptions() && is_array($form_values)) {
                foreach ($plugin_class->getPluginOptions() as $option_name => $option_value) {
                    if (isset($form_values[$option_name])) {
                        $option_value = $form_values[$option_name];
                    }
                    $this->setOption($option_name, $option_value);
                }
            }
        }
    }
}
