<?php
/**
 * RelatedProducts Merchandizing (Version 3.0.2)
 *
 * @author    Lineven
 * @copyright 2020 Lineven
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 * International Registered Trademark & Property of Lineven
 */

class LinevenRlpCrossSellingProductSearchProvider
{
    private $products_displayed;
    private $is_results_separated;
    private $sections;

    /**
     * LinevenRlpCrossSellingProductSearchProvider constructor.
     * @return void
     */
    public function __construct()
    {
        $this->products_displayed = array();
        $this->is_results_separated = false;
        $this->sections = array();
    }

    /**
     * Set is results separated
     * @param boolean $is_results_separated Is results separated
     * @return $this
     */
    public function setResultsSeparated($is_results_separated)
    {
        $this->is_results_separated = $is_results_separated;
        return $this;
    }

    /**
     * Is results separated.
     * @return int
     */
    public function isResultsSeparated()
    {
        return $this->is_results_separated;
    }

    /**
     * Set sections.
     * @param boolean $sections Sections
     * @return $this
     */
    public function setSections($sections)
    {
        $this->sections = $sections;
        return $this;
    }

    /**
     * Is section exists.
     * @param name $name Section name
     * @return $this
     */
    public function isSectionExists($name)
    {
        if (isset($this->sections[$name])) {
            return true;
        }
        return false;
    }

    /**
     * Get section max products value.
     * @param name $name Section name
     * @return int
     */
    public function getSectionMaxProducts($name)
    {
        if ($this->isSectionExists($name)) {
            // For old version of the module
            if (is_array($this->sections[$name]) && isset($this->sections[$name]['max_products'])) {
                return $this->sections[$name]['max_products'];
            } else {
                if (is_int($this->sections[$name])) {
                    return $this->sections[$name];
                }
            }
        }
        return false;
    }

    /**
     * Run query.
     * @param LinevenRlpProductSearchContext $context Context
     * @param LinevenRlpProductSearchQuery $query Query
     * @param boolean $run_plugin_query Run the query of the plugin
     * @return array
     */
    public function runQuery(
        LinevenRlpProductSearchContext $context,
        LinevenRlpProductSearchQuery $query,
        $run_plugin_query = true
    ) {
        $plugins = RelatedProducts::getPlugins();
        if ($query->isRandomPlugins() && !$this->isResultsSeparated()) {
            $this->sections = LinevenRlpTools::arrayShufflePreserve($this->sections);
        }
        if (count($plugins) && count($context->getVisibilities()) &&
            count($this->sections)) {
            $final_results = array();
            foreach ($this->sections as $plugin_code => $section) {
                if (isset($plugins[$plugin_code])) {
                    $plugin = $plugins[$plugin_code];
                    $query->setLimit($this->getSectionMaxProducts($plugin->code));
                    $plugin_class = LinevenRlpPluginObject::getStaticClass($plugin->classname);
                    if ($plugin_class != null) {
                        $plugin_options = array();
                        if ($plugin->hasOptions()) {
                            if (is_array($this->sections[$plugin->code])
                                && isset($this->sections[$plugin->code]['options'])
                                && count($this->sections[$plugin->code]['options'])) {
                                $plugin_options = $this->sections[$plugin->code]['options'];
                            } else {
                                $plugin_options = $plugin->getOptions();
                            }
                        }
                        $plugin_class->setPluginOptions($plugin_options);
                        $plugin_results = array();
                        if ($run_plugin_query) {
                            $plugin_results = $plugin_class->runQuery($context, $query);
                        }
                        if (count($plugin_results) || !$run_plugin_query) {
                            $count = 0;
                            if ($this->isResultsSeparated()) {
                                $final_results = array();
                            }
                            foreach ($plugin_results as $product) {
                                if ($count <= $query->getLimit()) {
                                    $id_product = 0;
                                    if ($plugin_class->getProductIdColumnName() != null) {
                                        $id_product = $product[$plugin_class->getProductIdColumnName()];
                                    } else {
                                        $id_product = $product;
                                    }
                                    if (!$plugin_class->isRelatedProductsMustVerified() ||
                                        ($plugin_class->isRelatedProductsMustVerified()
                                            && !in_array($id_product, $query->getRelatedProducts()))) {
                                        if ($this->addProductDisplayed($id_product, $plugin->code)) {
                                            $product['PLUGIN'] = $plugin->code;
                                            $final_results[$id_product] = $product;
                                            $count++;
                                        }
                                    }
                                } else {
                                    break;
                                }
                            }
                            if ($this->isResultsSeparated()) {
                                $products_result = new LinevenRlpProductSearchResult();
                                $products_result->setProductsFound($final_results);
                                $products_result->setProducts($this->getProducts(
                                    $context,
                                    $query,
                                    $final_results
                                ));
                                if ($products_result->getCount() || !$run_plugin_query) {
                                    $this->sections[$plugin->code]['max_products'] = $this->getSectionMaxProducts($plugin->code);
                                    $this->sections[$plugin->code]['plugin'] = $plugin;
                                    $this->sections[$plugin->code]['results'] = $products_result;
                                } else {
                                    unset($this->sections[$plugin->code]);
                                }
                            }
                        } else {
                            unset($this->sections[$plugin->code]);
                        }
                    }
                } else {
                    unset($this->sections[$plugin_code]);
                }
            }
            if (!$this->isResultsSeparated()) {
                if (count($final_results) || !$run_plugin_query) {
                    $products_result = new LinevenRlpProductSearchResult();
                    $products_result->setProductsFound($final_results);
                    $products_result->setProducts($this->getProducts(
                        $context,
                        $query,
                        $final_results
                    ));
                    return array('regroup' => array('SECTION' => array('results' => $products_result)));
                }
            } else {
                foreach ($this->sections as $key => $value) {
                    if ((!is_array($value))) {
                        unset($this->sections[$key]);
                    }
                }
                if (count($this->sections)) {
                    return array('separate' => $this->sections);
                }
            }
        }
        return array();
    }

    /**
     * Add product displayed.
     * @param int $id_product
     * @param string $plugin_code
     * @return void
     */
    private function addProductDisplayed($id_product, $plugin_code = null)
    {
        if ($id_product) {
            if ($this->isResultsSeparated() && $plugin_code != null) {
                if (isset($this->products_displayed[$plugin_code])) {
                    if (!in_array($id_product, $this->products_displayed[$plugin_code])) {
                        $this->products_displayed[$plugin_code][] = $id_product;
                        return true;
                    }
                } else {
                    $this->products_displayed[$plugin_code][] = $id_product;
                    return true;
                }
            } else {
                if (!in_array($id_product, $this->products_displayed)) {
                    $this->products_displayed[] = $id_product;
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Search products to display.
     *
     * @param LinevenRlpProductSearchContext $context Context
     * @param LinevenRlpProductSearchQuery $query Query
     * @param array $products Products
     * @return array
     */
    private function getProducts(
        LinevenRlpProductSearchContext $context,
        LinevenRlpProductSearchQuery $query,
        $products = null
    ) {
        if ($products && count($products)) {
            $in = implode(',', array_keys($products));
            $order_by = ' FIND_IN_SET(p.`id_product`, \''.$in.'\')';
            if ($query->getSortDisplayMethod() == RelatedProducts::$sort_display_name) {
                $order_by = ' pl.`name` '.$query->getSortDisplayWay();
            }
            $sql = 'SELECT p.*, product_shop.*, stock.out_of_stock, IFNULL(stock.quantity, 0) as quantity,
                    MAX(product_attribute_shop.id_product_attribute) id_product_attribute,
                    product_attribute_shop.minimal_quantity AS product_attribute_minimal_quantity, pl.`description`,
                    pl.`description_short`, pl.`available_now`,
                    pl.`available_later`, pl.`link_rewrite`, pl.`meta_description`, pl.`meta_keywords`,
                    pl.`meta_title`, pl.`name`, MAX(image_shop.`id_image`) id_image,
                    DATEDIFF(product_shop.`date_add`, DATE_SUB(NOW(),
                    INTERVAL '.(Validate::isUnsignedInt(Configuration::get('PS_NB_DAYS_NEW_PRODUCT'))
                    ? Configuration::get('PS_NB_DAYS_NEW_PRODUCT') : 20).'
                        DAY)) > 0 AS new, product_shop.price AS orderprice
                FROM `'._DB_PREFIX_.'product` p
                '.Shop::addSqlAssociation('product', 'p').'
                LEFT JOIN `'._DB_PREFIX_.'product_attribute` pa
                ON (p.`id_product` = pa.`id_product`)
                '.Shop::addSqlAssociation('product_attribute', 'pa', false, 'product_attribute_shop.`default_on` = 1').'
                '.Product::sqlStock('p', 'product_attribute_shop', false, Context::getContext()->shop).'
                LEFT JOIN `'._DB_PREFIX_.'product_lang` pl
                    ON (p.`id_product` = pl.`id_product`
                    AND pl.`id_lang` = '.(int)$context->getIdLang().Shop::addSqlRestrictionOnLang('pl').')
                LEFT JOIN `'._DB_PREFIX_.'image` i
                    ON (i.`id_product` = p.`id_product`)'.
                Shop::addSqlAssociation('image', 'i', false, 'image_shop.cover=1').'
                WHERE product_shop.`id_shop` = '.(int)$context->getIdShop().'
                    AND product_shop.`visibility` IN ('.$context->getVisibilitiesSql().')
                    AND product_shop.`active` = 1
                    AND product_shop.`available_for_order` = 1
                    AND p.`id_product` in ('.$in.')
                GROUP BY product_shop.id_product
                ORDER BY '.$order_by;
            $results = Db::getInstance()->executeS($sql);
            if ($query->getSortDisplayMethod() == RelatedProducts::$sort_display_price) {
                Tools::orderbyPrice($results, $query->getSortDisplayWay());
            }
            if ($query->getSortDisplayMethod() == RelatedProducts::$sort_display_random) {
                shuffle($results);
            }
            return $results;
        }
        return array();
    }
}
