/**
 * Script Module.
 *
 * RelatedProducts Merchandizing (Version 3.0.2)
 *
 * @author Lineven
 * @copyright 2020 Lineven
 * @license    http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 * International Registered Trademark & Property of Lineven
 */

var LinevenRlpModule = {
	
	/**
	 * Associations.
	 */
	Associations: {
        /**
         * List.
         */
        List: {

            positions: {first: 0, last: 0},

            /**
             * Prepare content.
             */
            prepareContent: function () {
                LinevenRlpModule.Associations.List.initDndOnList();
            },

            /**
             * Drag & Drop initialization.
             */
            initDndOnList: function() {
                $("#table-lrlp-associations-list").tableDnD({
                    onDragStart: function(table, row) {
                        $('.lrlp-associations-list-item').animate({opacity: 0.2}, 200);
                        $('#'+$(row).attr('id')).animate({opacity: 1}, 200);
                    },
                    onDrop: function(table, row) {
                        $('.lrlp-associations-list-item').animate({opacity: 1}, 200);
                        LinevenRlpModule.Associations.List.dndSetPositions();
                    }
                });
                LinevenRlpModule.Associations.List.initPositions();
            },

            /**
             * Get Row id for dnd.
             */
            dndGetRowId: function(row_tag_id) {
                return row_tag_id.substr(4, (row_tag_id.length - 6));
            },

            /**
             * Init positions.
             */
            initPositions: function () {
                $('#table-lrlp-associations-list>tbody tr.lrlp-associations-list-item').each(function(index) {
                    if (index == 0) {
                        LinevenRlpModule.Associations.List.positions.first = parseInt($(this).find('.lineven-helper-list-position').html().trim());
                    }
                    LinevenRlpModule.Associations.List.positions.last = parseInt($(this).find('.lineven-helper-list-position').html().trim());
                });
            },

            /**
             * Set positions.
             */
            dndSetPositions: function () {
                var a_positions = [];
                var position = LinevenRlpModule.Associations.List.positions.first;
                $('#table-lrlp-associations-list>tbody tr.lrlp-associations-list-item').each(function() {
                    $(this).find('.lineven-helper-list-position').html(position);
                    a_positions.push(LinevenRlpModule.Associations.List.dndGetRowId($(this).attr('id')));
                    position = position + 1;
                });
                // Send
                $.ajax({
                    url: LinevenRlp.getActionUri(),
                    type: 'post',
                    data: {
                        relatedproducts_controller: 'Associations',
                        relatedproducts_action: 'order',
                        positions: a_positions.join(','),
                        first_position: LinevenRlpModule.Associations.List.positions.first,
                        last_position: LinevenRlpModule.Associations.List.positions.last
                    },
                    success: function(data)
                    {
                    }
                });
            }
        },

        /**
		 * Edit.
         */
		Edit: {

            /**
             * Prepare content to add / edit association.
             */
            prepareContent: function() {
                // Prepare display product search
                $('#display_product_content_input').autocomplete(LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Associations&'+LinevenRlp.module_name+'_action=getProducts', {
                    minChars: 1,
                    autoFill: true,
                    max: 40,
                    matchContains: true,
                    mustMatch: true,
                    scroll: false,
                    cacheLength: 0,
                    formatItem: function(item) {
                        return '<div>' +
                            '<table><tr>' +
                            '<td style="padding-right: 10px">'+ item[3] +'</td>' +
                            '<td>' + item[1] + '<br />Ref: ' + item[2] + '</td>' +
                            '</tr></table></div>';
                    }
                }).result(LinevenRlpModule.Associations.Edit.addDisplayProductToAssociation);
                if ($('#product_id').val() != '') {
                    LinevenRlpModule.Associations.Edit.previewDisplayProduct($('#product_id').val());
                }
                // Prepare selection product search
                $('#selection_product_content_input').autocomplete(LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Associations&'+LinevenRlp.module_name+'_action=getProducts', {
                    minChars: 1,
                    autoFill: true,
                    max: 40,
                    matchContains: true,
                    mustMatch: true,
                    scroll: false,
                    cacheLength: 0,
                    formatItem: function(item) {
                        return '<div>' +
                            '<table><tr>' +
                            '<td style="padding-right: 10px">'+ item[3] +'</td>' +
                            '<td>' + item[1] + '<br />Ref: ' + item[2] + '</td>' +
                            '</tr></table></div>';
                    }
                }).result(LinevenRlpModule.Associations.Edit.addSelectionProductToAssociation);
                if ($('#related_product_id').val() != '') {
                    LinevenRlpModule.Associations.Edit.previewSelectionProduct($('#related_product_id').val());
                }
                // On association type changed
                LinevenRlpModule.Associations.Edit.doOnAssociationTypeChange();
                $('#association_type').change(function() {
                    LinevenRlpModule.Associations.Edit.doOnAssociationTypeChange();
                });

                // On association display choice changed
                LinevenRlpModule.Associations.Edit.doOnDisplayChoiceChange();
                $('#display_choice').change(function() {
                    LinevenRlpModule.Associations.Edit.doOnDisplayChoiceChange();
                });
            },

            /**
             * Add display product to association.
             */
            addDisplayProductToAssociation: function(event, data, formatted) {
                if (data == null)
                    return false;
                var id_product = parseInt(data[0]);
                var productName = data[1];
                $('#display_product_content_input').val('');
                $('#product_id').val(id_product);
                LinevenRlpModule.Associations.Edit.previewDisplayProduct(id_product);
            },

            /**
             * Add selection product to association.
             */
            addSelectionProductToAssociation: function(event, data, formatted) {
                if (data == null)
                    return false;
                var id_product = parseInt(data[0]);
                var productName = data[1];
                $('#selection_product_content_input').val('');
                $('#related_product_id').val(id_product);
                LinevenRlpModule.Associations.Edit.previewSelectionProduct(id_product);
            },

            /**
             * Preview for display product.
             */
            previewDisplayProduct: function(id_product) {
                $.ajax({
                    type: "POST",
                    dataType: "json",
                    url: LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Associations&'+LinevenRlp.module_name+'_action=previewProduct&for=id_product_displayed&product_object=product_displayed',
                    data: {
                        id_product_displayed: id_product
                    },
                    cache: false,
                    success: function(jsonData, textStatus, jqXHR) {
                        if (jsonData != undefined && jsonData.html != undefined) {
                            $('#display-product-preview-table>tbody').html(jsonData.html);
                        }
                    },
                    error: function(XMLHttpRequest, textStatus, errorThrown) {
                    },
                    complete: function(jqXHR, textStatus) {
                    }
                });
            },

            /**
             * Preview for product selection.
             */
            previewSelectionProduct: function(id_product) {
                $.ajax({
                    type: "POST",
                    dataType: "json",
                    url: LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Associations&'+LinevenRlp.module_name+'_action=previewProduct&for=id_product_selection&product_object=product_selection',
                    data: {
                        id_product_selection: id_product
                    },
                    cache: false,
                    success: function(jsonData, textStatus, jqXHR) {
                        if (jsonData != undefined && jsonData.html != undefined) {
                            $('#selection-product-preview-table>tbody').html(jsonData.html);
                        }
                    },
                    error: function(XMLHttpRequest, textStatus, errorThrown) {
                    },
                    complete: function(jqXHR, textStatus) {
                    }
                });
            },

            /**
             * Remove selection product.
             */
            removeSelectionProduct: function(id_product) {
                $('#related_product_id').val('');
                $('#selection-product-preview-table>tbody').html('');
            },

            /**
             * Remove display product.
             */
            removeDisplayProduct: function(id_product) {
                $('#product_id').val('');
                $('#display-product-preview-table>tbody').html('');
            },

            /**
             * Do on association type changement.
             */
            doOnAssociationTypeChange: function() {
                if ($('#association_type').val() == 'ALL') {
                    $('div[id^="fieldset_selection_rule_category"]').fadeOut();
                    $('div[id^="fieldset_selection_rule_keywords"]').fadeOut();
                    $('div[id^="fieldset_selection_rule_product"]').fadeOut();
                }
                if ($('#association_type').val() == 'CATEGORY') {
                    $('div[id^="fieldset_selection_rule_category"]').show();
                    $('div[id^="fieldset_selection_rule_keywords"]').fadeOut();
                    $('div[id^="fieldset_selection_rule_product"]').fadeOut();
                }
                if ($('#association_type').val() == 'KEYWORDS') {
                    $('div[id^="fieldset_selection_rule_category"]').fadeOut();
                    $('div[id^="fieldset_selection_rule_keywords"]').show();
                    $('div[id^="fieldset_selection_rule_product"]').fadeOut();
                }
                if ($('#association_type').val() == 'PRODUCT') {
                    $('div[id^="fieldset_selection_rule_category"]').fadeOut();
                    $('div[id^="fieldset_selection_rule_keywords"]').fadeOut();
                    $('div[id^="fieldset_selection_rule_product"]').show();
                }
                if ($('#association_type').val() == 'PRODUCT' && $('#display_choice').val() == 'PRODUCT') {
                    $('div[id^="fieldset_reciprocity"]').show();
                } else {
                    $('div[id^="fieldset_reciprocity"]').fadeOut();
                    $('#is_reciprocity_on').prop('checked', false);
                }
            },

            /**
             * Do on association display choice changement.
             */
            doOnDisplayChoiceChange: function() {
                if ($('#display_choice').val() == 'CATEGORY') {
                    $('div[id^="fieldset_display_category"]').show();
                    $('div[id^="fieldset_display_keywords"]').fadeOut();
                    $('div[id^="fieldset_display_product"]').fadeOut();
                }
                if ($('#display_choice').val() == 'PRODUCT') {
                    $('div[id^="fieldset_display_category"]').fadeOut();
                    $('div[id^="fieldset_display_keywords"]').fadeOut();
                    $('div[id^="fieldset_display_product"]').show();
                }
                if ($('#display_choice').val() == 'KEYWORDS') {
                    $('div[id^="fieldset_display_category"]').fadeOut();
                    $('div[id^="fieldset_display_keywords"]').show();
                    $('div[id^="fieldset_display_product"]').fadeOut();
                }
                if ($('#association_type').val() == 'PRODUCT' && $('#display_choice').val() == 'PRODUCT') {
                    $('div[id^="fieldset_reciprocity"]').show();
                } else {
                    $('div[id^="fieldset_reciprocity"]').fadeOut();
                    $('#is_reciprocity_on').prop('checked', false);
                }
            },

            /**
             * Save association.
             * @param int $id Association Id
             * @param boolean $stay Stay
             */
            doOnClickAssociationSave: function (id, stay) {
                if ($('#module_form').length) {
                    if (stay) {
                        $('#module_form').prepend('<input type="hidden" name="save_stay" value="1">');
                    }
                    $('#module_form').submit();
                }
            },

            /**
             * Cancel association.
             */
            doOnClickAssociationCancel: function () {
                window.location = LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Associations&'+LinevenRlp.module_name+'_action=list';
            }
		}
	},

    /**
     * Plugins.
     */
    Plugins: {
        /**
         * List.
         */
        List: {

            positions: {first: 0, last: 0},

            /**
             * Prepare content.
             */
            prepareContent: function () {
                LinevenRlpModule.Plugins.List.initDndOnList();
            },

            /**
             * Drag & Drop initialization.
             */
            initDndOnList: function () {
                $("#table-lrlp-plugins-list").tableDnD({
                    onDragStart: function (table, row) {
                        $('.lrlp-plugins-list-item').animate({opacity: 0.2}, 200);
                        $('#' + $(row).attr('id')).animate({opacity: 1}, 200);
                    },
                    onDrop: function (table, row) {
                        $('.lrlp-plugins-list-item').animate({opacity: 1}, 200);
                        LinevenRlpModule.Plugins.List.dndSetPositions();
                    }
                });
                LinevenRlpModule.Plugins.List.initPositions();
            },

            /**
             * Get Row id for dnd.
             */
            dndGetRowId: function (row_tag_id) {
                return row_tag_id.substr(4, (row_tag_id.length - 6));
            },

            /**
             * Init positions.
             */
            initPositions: function () {
                $('#table-lrlp-plugins-list>tbody tr.lrlp-plugins-list-item').each(function (index) {
                    if (index == 0) {
                        LinevenRlpModule.Plugins.List.positions.first = parseInt($(this).find('.lineven-helper-list-position').html().trim());
                    }
                    LinevenRlpModule.Plugins.List.positions.last = parseInt($(this).find('.lineven-helper-list-position').html().trim());
                });
            },

            /**
             * Set positions.
             */
            dndSetPositions: function () {
                var a_positions = [];
                var position = LinevenRlpModule.Plugins.List.positions.first;
                $('#table-lrlp-plugins-list>tbody tr.lrlp-plugins-list-item').each(function () {
                    $(this).find('.lineven-helper-list-position').html(position);
                    a_positions.push(LinevenRlpModule.Plugins.List.dndGetRowId($(this).attr('id')));
                    position = position + 1;
                });

                // Send
                $.ajax({
                    url: LinevenRlp.getActionUri(),
                    type: 'post',
                    data: {
                        relatedproducts_controller: 'Plugins',
                        relatedproducts_action: 'order',
                        positions: a_positions.join(','),
                        first_position: LinevenRlpModule.Plugins.List.positions.first,
                        last_position: LinevenRlpModule.Plugins.List.positions.last
                    },
                    success: function (data) {
                    }
                });
            }
        },
    },


    /**
     * Hooks.
     */
    Hooks: {
        /**
         * Prepare content.
         */
        prepareContent: function () {
            LinevenRlpModule.Hooks.toggleSections();
            $('input[name="separate_display"]').on('change', function (event) {
                LinevenRlpModule.Hooks.toggleSections();
            });
            if ($('#hook_to_use').length) {
                LinevenRlpModule.Hooks.onHookUsedChange();
                $('#hook_to_use').on('change', function (event) {
                    LinevenRlpModule.Hooks.onHookUsedChange();
                });
            }
            if ($('#template_to_use').length) {
                LinevenRlpModule.Hooks.onTemplateChange();
                $('#template_to_use').on('change', function (event) {
                    LinevenRlpModule.Hooks.onTemplateChange();
                });
            }
            if ($('#refresh_method').length) {
                LinevenRlpModule.Hooks.onRefreshMethodChange();
                $('#refresh_method').on('change', function (event) {
                    LinevenRlpModule.Hooks.onRefreshMethodChange();
                });
            }
            LinevenRlpModule.Hooks.onSortChange();
            $('#sort_display_method').on('change', function (event) {
                LinevenRlpModule.Hooks.onSortChange();
            });
            // Prepare plugin content
            LinevenRlpModule.Hooks.preparePluginContent();

            // Hide confirm/error message
            LinevenRlpModule.Hooks.hideSpecificNotifications();
        },

        /**
         * Hide specific notification message.
         */
        hideSpecificNotifications: function () {
            $('div[id^="lineven-template-content-hooks-"] .alert-success').closest('.form-group').hide();
            $('div[id^="lineven-template-content-hooks-"] .alert-danger').closest('.form-group').hide();
        },

        /**
         * Prepare plugin content.
         */
        preparePluginContent: function () {
           $('.hook-plugin-add').on('click', function (event) {
                LinevenRlpModule.Hooks.addPlugin($(this).attr('data-plugin-id'));
            });
            $('#hook-plugins-list').on('click', '.hook-plugin-delete', function( event ) {
                LinevenRlpModule.Hooks.deletePlugin($(this).attr('data-plugin-id'));
            });
            LinevenRlpModule.Hooks.togglePluginDeleteButton();
            LinevenRlpModule.Hooks.initDndOnPlugins();
        },

        /**
         * On hook used change.
         */
        onHookUsedChange: function () {
            LinevenRlpModule.Hooks.hideSpecificNotifications();
            var hook_to_use = $('#hook_to_use').val();
            if (hook_to_use == 'SPECIFIC') {
                $('#specific_hook_to_use').closest('.form-group').fadeIn();
            } else {
                $('#specific_hook_to_use').closest('.form-group').fadeOut();
                $('#specific_hook_to_use').val('');
            }
        },

        /**
         * On template change.
         */
        onTemplateChange: function () {
            var template_to_use = $('#template_to_use').val();
            if (template_to_use == 'THUMBNAILS') {
                $('div[id^="fieldset_display_thumbnails_settings"]').fadeIn();
            } else {
                $('div[id^="fieldset_display_thumbnails_settings"]').fadeOut();
            }
            if (template_to_use == 'THEME' || template_to_use.indexOf('_theme', 0) > 0) {
                $('#image_type').closest('.form-group').fadeOut();
                $('#display_price_on').closest('.form-group').fadeOut();
                $('#display_cart_on').closest('.form-group').fadeOut();
                $('#display_new_product_on').closest('.form-group').fadeOut();
                $('#display_reduction_on').closest('.form-group').fadeOut();
            } else {
                $('#image_type').closest('.form-group').fadeIn();
                $('#display_price_on').closest('.form-group').fadeIn();
                $('#display_cart_on').closest('.form-group').fadeIn();
                $('#display_new_product_on').closest('.form-group').fadeIn();
                $('#display_reduction_on').closest('.form-group').fadeIn();
            }
        },

        /**
         * On refresh method change.
         */
        onRefreshMethodChange: function () {
            var refresh_method = $('#refresh_method').val();
            if (refresh_method != 'NOTHING') {
                $('#refresh_delay').closest('.form-group').fadeIn();
            } else {
                $('#refresh_delay').closest('.form-group').fadeOut();
            }
        },

        /**
         * On sort change.
         */
        onSortChange: function () {
            if ($('#sort_display_method').val() == 'NAME' || $('#sort_display_method').val() == 'PRICE') {
                $('#sort_display_way').closest('.form-group').fadeIn();
            } else {
                $('#sort_display_way').closest('.form-group').fadeOut();
            }
        },

        /**
         * Toogle sections.
         */
        toggleSections: function () {
            if ($('div[id^="fieldset_sections_activation_"]').length) {
                if ($('#separate_display_on').length) {
                    if ($('#separate_display_on').is(':checked')) {
                        $('div[id^="fieldset_sections_activation_"] .form-group:not(:first)').fadeOut();
                        $('input[id^="hook_max_products_plugin_"]').closest('.form-group').fadeIn();
                    } else {
                        $('div[id^="fieldset_sections_activation_"] .form-group:not(:first)').fadeIn();
                        $('input[id^="hook_max_products_plugin_"]').closest('.form-group').fadeOut();
                    }
                } else {
                    $('div[id^="fieldset_sections_activation_"] .form-group:not(:first)').fadeIn();
                    $('input[id^="hook_max_products_plugin_"]').closest('.form-group').fadeOut();
                }
            }
        },

        /**
         * Link to a specific hook
         */
        linkToSpecificHook: function ()
        {
            LinevenRlpModule.Hooks.hideSpecificNotifications();
            if ($('#specific_hook_to_use').val().trim() != '') {
                $.ajax({
                    type: "POST",
                    dataType: "json",
                    url: LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Hooks&'+LinevenRlp.module_name+'_action=linkToHook&no_layout',
                    data: {
                        hook_name: $('#specific_hook_to_use').val().trim(),

                    },
                    cache: false,
                    success: function(jsonData, textStatus, jqXHR) {
                        if (jsonData != undefined && jsonData.return == 1) {
                            $('div[id^="lineven-template-content-hooks-"] .alert-success').closest('.form-group').show();
                        } else {
                            $('div[id^="lineven-template-content-hooks-"] .alert-danger').closest('.form-group').show();
                        }
                    },
                    error: function(XMLHttpRequest, textStatus, errorThrown) {
                        $('div[id^="lineven-template-content-hooks-"] .alert-danger').closest('.form-group').show();
                    },
                    complete: function(jqXHR, textStatus) {}
                });
            }
        },

        /**
         * Calculate plugin position.
         */
        calculatePluginPosition: function(index) {
            $('.hook-panel-plugin-badge-position').each(function(index) {
                $(this).html(index+1);
            });
        },

        /**
         * Delete plugin.
         * @param int id_plugin Plugin id
         */
        deletePlugin: function (id_plugin)
        {
            var plugins_positions = $('#hook_plugins_positions').val().split(',');
            plugins_positions.splice( $.inArray(id_plugin, plugins_positions), 1 );
            $('#hook_plugins_positions').val(plugins_positions.join(','));
            $('#hook-panel-plugin_'+id_plugin).remove();
            $('.hook-plugin-add-button-'+id_plugin).removeClass('disabled');
            LinevenRlpModule.Hooks.togglePluginDeleteButton();
            LinevenRlpModule.Hooks.calculatePluginPosition();
        },

        /**
         * Plugin Drag & Drop initialization.
         */
        initDndOnPlugins: function() {
            $('#hook-plugins-list').sortable("destroy");
            $('#hook-plugins-list').sortable({
                handle: '.panel-heading',
                forcePlaceholderSize: true,
                placeholder: "placeholder"
            });
            $("#hook-plugins-list").disableSelection();
            $("#hook-plugins-list .panel-heading").on("mousedown",function(event,ui){
                $('#hook-plugins-list .panel-heading').css('cursor', 'move');
            });
            $("#hook-plugins-list .panel-heading").on("mouseup",function(event,ui){
                $('#hook-plugins-list .panel-heading').css('cursor', 'auto');
            });
            $('#hook-plugins-list').on('sortupdate',function() {
                $('#hook-plugins-list .panel-heading').css('cursor', 'auto');
                $('#hook_plugins_positions').val('');
                $('.hook-panel-plugin').each(function(index) {
                    if (index != 0) {
                        $('#hook_plugins_positions').val($('#hook_plugins_positions').val()+',');
                    }
                    $('#hook_plugins_positions').val($('#hook_plugins_positions').val()+$(this).attr('data-plugin-id'));
                    LinevenRlpModule.Hooks.calculatePluginPosition();
                });
            });
        },

        /**
         * Add plugin.
         * @param int id_plugin Plugin id
         */
        addPlugin: function (id_plugin) {
            $.ajax({
                type: "POST",
                dataType: "json",
                url: LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Hooks&'+LinevenRlp.module_name+'_action=addPlugin&no_layout',
                data: {
                    id_plugin: id_plugin
                },
                cache: false,
                success: function(jsonData, textStatus, jqXHR) {
                    if (jsonData != undefined && jsonData.html != '') {
                        $(jsonData.html).insertBefore($('#hook_plugins_positions'));
                        var plugins_positions = $('#hook_plugins_positions').val().split(',');
                        plugins_positions.push(id_plugin);
                        $('.hook-plugin-add-button-'+id_plugin).addClass('disabled');
                        $('#hook_plugins_positions').val(plugins_positions.join(','));
                        if ($('#hook_plugins_positions').val().charAt(0) == ',') {
                            $('#hook_plugins_positions').val($('#hook_plugins_positions').val().substr(1));
                        }
                        LinevenRlpModule.Hooks.toggleSections();
                        $([document.documentElement, document.body]).animate({
                            scrollTop: $('#hook-panel-plugin_'+id_plugin).offset().top
                        }, 200);
                        LinevenRlpModule.Hooks.togglePluginDeleteButton();
                        LinevenRlpModule.Hooks.initDndOnPlugins();
                        LinevenRlpModule.Hooks.calculatePluginPosition();
                    }
                },
                error: function(XMLHttpRequest, textStatus, errorThrown) {},
                complete: function(jqXHR, textStatus) {}
            });
        },

        /**
         * Toggle plugin delete button.
         */
        togglePluginDeleteButton: function () {
            if ($('.hook-panel-plugin').length > 1) {
                $('.hook-panel-plugin').first().find('.hook-plugin-delete').closest('a').show();
            } else {
                $('.hook-panel-plugin').first().find('.hook-plugin-delete').closest('a').hide();
            }
        },

        /**
         * Get id plugin in id element.
         * @param string id_element id Element
         * @param string pattern Pattern
         */
        getIdPluginInIdElement: function (id_element, pattern) {
            return id_element.substring(pattern.length);
        }
    },

    /**
	 * Design.
	 */
	Design: {
	
		/**
		 * Prepare Design.
		 */
		prepare: function () {
			LinevenRlpModule.Design.toggleMode();
			$('#active_default_design_on').change(function() {
				LinevenRlpModule.Design.toggleMode();
			});
			$('#active_default_design_off').change(function() {
				LinevenRlpModule.Design.toggleMode();
			});
			$('#design_display_mode').change(function() {
				LinevenRlpModule.Design.load();
			});
			$('#reload_styles').click(function() {
				LinevenRlpModule.Design.load(true);
			});
		},
		
		/**
		 * Toogle mode.
		 */
		toggleMode: function () {
			var useDefaultValue = false;
			if ($('#active_default_design_on').prop('checked') == true) {
				useDefaultValue = true;
			}
			if (useDefaultValue) {
				$('#design_display_mode').prop('disabled', true);
				$('#design_textarea').prop('disabled', true);
			} else {
				$('#design_display_mode').prop('disabled', false);
				$('#design_textarea').prop('disabled', false);
			}
			LinevenRlpModule.Design.load(false);
		},
		
		/**
		 * Load design.
		 * @param boolean default_style Is default style
		 */
		load: function (default_style) {
			// Send
			$.ajax({
				url: LinevenRlp.getActionUri(),
				type: 'POST',
				data: {
					relatedproducts_controller:'Tools',
					relatedproducts_action:'loadDesign',
					design_display_mode: $('#design_display_mode').val(),
					default_style: default_style
				},
				success: function(data)
				{
					$('#design_textarea').val(data);
				}
			});
		},
	},

	/**
	 * Catalog.
	 */
	Catalog: {
		bloodhound: null,
		
		/**
		 * Hide messages.
		 */
		hideMessages: function() {
			$('#rlp-catalog-product-confirmation-add-message').hide();
			$('#rlp-catalog-product-confirmation-delete-message').hide();
			$('#rlp-catalog-product-error-message').hide();
		},
	
		/**
		 * Prepare content to add / edit association from catalog.
		 */
        prepareContent: function() {
			LinevenRlpModule.Catalog.hideMessages();
			if (typeof(lineven_rlp_catalog_product) != 'undefined') { // For PS 1.7 catalog
				if (LinevenRlpModule.Catalog.bloodhound == null) {
					LinevenRlpModule.Catalog.initBloodhound();
				}
				$('#rlp_catalog_product_search_content_input').typeahead({
					limit: 20,
					minLength: 2,
					highlight: true,
					hint: false
				}, {
					display: 'id',
					source: LinevenRlpModule.Catalog.bloodhound,
					limit: 30,
					templates: {
						suggestion: function(item){
							return '<div>' +
								   '<table><tr>' +
								   '<td style="padding-right: 10px">'+ item.image +'</td>' +
								   '<td>' + item.name + '<br />REF: ' + item.ref + '</td>' +
								   '</tr></table></div>'
						}
					}
				}).bind('typeahead:select', function(ev, suggestion) {
					LinevenRlpModule.Catalog.addDisplayProductToAssociation(ev, suggestion, null)
					$(ev.target).val('');
				}).bind('typeahead:close', function(ev) {
					$(ev.target).val('');
				});
			} else {
				// Prepare display product search
				$('#rlp_catalog_product_search_content_input').autocomplete(LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Catalog&'+LinevenRlp.module_name+'_action=getProducts', {
					minChars: 1,
					autoFill: true,
					max: 40,
					matchContains: true,
					mustMatch: true,
					scroll: false,
					cacheLength: 0,
					formatItem: function(item) {
						return '<div>' +
						   '<table><tr>' +
						   '<td style="padding-right: 10px">'+ item[3] +'</td>' +
						   '<td>' + item[1] + '<br />Ref: ' + item[2] + '</td>' +
						   '</tr></table></div>';
					}
				}).result(LinevenRlpModule.Catalog.addDisplayProductToAssociation);
			}
		},
		
		/**
		 * Init bloodhound.
		 */
		initBloodhound: function() {
			LinevenRlpModule.Catalog.bloodhound = new Bloodhound({
				datumTokenizer: Bloodhound.tokenizers.whitespace,
				queryTokenizer: Bloodhound.tokenizers.whitespace,
				identify: function(obj) {
					return obj.id;
				},
				remote: {
					url: LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Catalog&'+LinevenRlp.module_name+'_action=getProducts&q=%QUERY',
					cache: false,
					wildcard: '%QUERY',
					transform: function(response){
						var newResponse = [];
						if(!response){
							return newResponse;
						}
						$.each(response, function(key, item){
						   newResponse.push(item);
						});
						return newResponse;
					}
				}
			});
		},

		/**
		 * Add display product to association.
		 */
		addDisplayProductToAssociation: function(event, data, formatted) {
			if (data == null) {
				return false;
			}
			var related_id_product = $('#id_related_product').val();
            var displayed_id_product = data.id;
            var displayed_name_product = data.name;
			$('#rlp_catalog_product_search_content_input').val('');
			LinevenRlpModule.Catalog.hideMessages();
			$.ajax({
				type: "POST",
				dataType: "json",
				url: LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Catalog&'+LinevenRlp.module_name+'_action=quicklyAddProduct',
				data: {
					id_related_product: related_id_product,
					id_displayed_product: displayed_id_product
				},
				cache: false,
				success: function(jsonData, textStatus, jqXHR) {
					if (jsonData != undefined && jsonData.result == true) {
						$('#rlp-catalog-product-confirmation-add-message').show();
						if (jsonData.html != undefined) {
							$('#rlp-catalog-product-list>tbody').html(jsonData.html);
						}
					} else {
						$('#rlp-catalog-product-error-message').show();			
					}
				},
				error: function(XMLHttpRequest, textStatus, errorThrown) {
					$('#rlp-catalog-product-error-message').show();	
				},
				complete: function(jqXHR, textStatus) {
				}
			});
		},
		
		/**
		 * Remove cart product.
		 */
		removeDisplayProduct: function(id_association) {
			LinevenRlpModule.Catalog.hideMessages();
			var related_id_product = $('#id_related_product').val();
			$.ajax({
				type: "POST",
				dataType: "json",
				url: LinevenRlp.getActionUri()+'&'+LinevenRlp.module_name+'_controller=Catalog&'+LinevenRlp.module_name+'_action=deleteAssociation',
				data: {
					id_related_product: related_id_product,
					id_association: id_association
				},
				cache: false,
				success: function(jsonData, textStatus, jqXHR) {
					if (jsonData != undefined && jsonData.result == true) {
						$('#rlp-catalog-product-confirmation-delete-message').show();
						if (jsonData.html != undefined) {
							$('#rlp-catalog-product-list>tbody').html(jsonData.html);
						}
					} else {
						$('#rlp-catalog-product-error-message').show();	
					}
				},
				error: function(XMLHttpRequest, textStatus, errorThrown) {
					$('#rlp-catalog-product-error-message').show();	
				},
				complete: function(jqXHR, textStatus) {
				}
			});
		}
	}
};

$(document).ready(function() {
    if ($('#lineven-template-content-plugins-list').length) {
        LinevenRlpModule.Plugins.List.prepareContent();
    }
    if ($('div[id^="lineven-template-content-hooks-on"]').length) {
        LinevenRlpModule.Hooks.prepareContent();
    }
});