'use strict'
const path = require('path')
const ip = require('ip')
const glob = require('glob')
const UglifyJsPlugin = require('uglifyjs-webpack-plugin')
const ExtractTextPlugin = require('extract-text-webpack-plugin')
const ManifestPlugin = require('webpack-manifest-plugin')
const CleanWebpackPlugin = require('clean-webpack-plugin')
const BrowserSyncPlugin = require('browser-sync-webpack-plugin')

const dev = process.env.NODE_ENV === 'dev' || process.env.NODE_ENV === 'start'
const start = process.env.NODE_ENV === 'start'

const projectHost = 'afilor.local'

let extractSass = new ExtractTextPlugin({
  filename: dev ? 'css/[name].css' : 'css/[name].[contenthash:8].css',
  disable: start
})

let cssLoaders = [{
  loader: 'css-loader',
  options: {
    importLoaders: 1,
    sourceMap: dev,
    minimize: !dev,
    url: true
  }
}]

if (!dev) {
  cssLoaders.push({
    loader: 'postcss-loader',
    options: {
      ident: 'postcss',
      plugins: (loader) => [
        require('autoprefixer')({
          browsers: ['last 5 versions', 'ie > 10']
        })
      ]
    }
  })
}

let entriesImg = glob.sync('./assets/images/*')
let entries = glob.sync('./assets/svg/*.svg')
entries.push('./assets/css/theme.scss')
entries.push('./assets/js/theme.js')
entries = entries.concat(entriesImg)

let config = {
  entry: {
    'theme': entries
  },
  output: {
    path: path.resolve(__dirname, '../assets'),
    filename: dev ? 'js/[name].js' : 'js/[name].[hash:8].js',
    publicPath: (start ? 'http://' + ip.address() + ':9000' : '') + '/assets/'
  },
  devServer: {
    contentBase: path.resolve(__dirname),
    compress: false,
    host: '0.0.0.0',
    port: 9000,
    useLocalIp: false,
    bonjour: false,
    open: false,
    overlay: {
      warnings: true,
      errors: true
    },
    headers: {
      'Access-Control-Allow-Origin': '*',
      'Access-Control-Allow-Methods': 'GET, POST, PUT, DELETE, PATCH, OPTIONS',
      'Access-Control-Allow-Headers': 'X-Requested-With, content-type, Authorization'
    }
  },
  resolve: {
    alias: {
      '@css': path.resolve(__dirname, 'assets/css'),
      '@js': path.resolve(__dirname, 'assets/js'),
      '@font': path.resolve(__dirname, 'assets/fonts'),
      '@img': path.resolve(__dirname, 'assets/images'),
      '@svg': path.resolve(__dirname, 'assets/svg')
    }
  },
  devtool: dev ? 'eval-source-map' : false,
  module: {
    rules: [
      {
        test: /\.js$/,
        exclude: /(node_modules|bower_components|tools)/,
        use: [
          {
            loader: 'babel-loader'
          },
          {
            loader: 'eslint-loader'
          }
        ]
      },
      {
        test: /\.scss$/,
        use: extractSass.extract({
          use: [
            ...cssLoaders,
            {
              loader: 'sass-loader',
              options: {
                sourceMap: dev
              }
            }
          ],
          fallback: 'style-loader'
        })
      },
      {
        test: /\.(woff2?|eot|ttf|otf)(\?.*)?$/,
        use: [
          {
            loader: 'file-loader',
            options: {
              outputPath: 'fonts/',
              name: '[name].[hash:8].[ext]'
            }
          }
        ]
      },
      {
        test: /\.(svg)$/,
        use: [
          {
            loader: 'file-loader',
            options: {
              outputPath: 'svg/',
              name: '[name].[ext]'
            }
          },
          {
            loader: 'img-loader',
            options: {
              svgo: {
                plugins: [
                  { removeXMLProcInst: true },
                  { removeRasterImages: true },
                  { removeXMLNS: false },
                  { cleanupAttrs: true },
                  { removeDoctype: true },
                  { removeEmptyContainers: true },
                  { collapseGroups: false },
                  { cleanupIDs: true },
                  { removeDimensions: true },
                  { mergePaths: false }
                ]
              }
            }
          }
        ]
      },
      {
        test: /\.(png|jpe?g|gif)$/,
        use: [
          {
            loader: 'file-loader',
            options: {
              outputPath: 'images/',
              name: start ? '[name].[ext]' : '[name].[hash:8].[ext]'
            }
          },
          {
            loader: 'img-loader',
            options: {
              enabled: !dev
            }
          }
        ]
      }
    ]
  },
  plugins: [
    extractSass,
    new CleanWebpackPlugin(['../assets'], {
      dry: false
    }),
    new ManifestPlugin()
  ],
  externals: {
    prestashop: 'prestashop'
  }
}

if (!dev) {
  config.plugins.push(
    new UglifyJsPlugin()
  )
}
if (start) {
  config.plugins.push(
    new BrowserSyncPlugin({
      proxy: 'http://' + projectHost + '',
      reloadDelay: 0
    }, {
      reload: false
    })
  )
}

module.exports = config
